/*******************************************************************************
*   Filename:       GenericBroadcaster_Main.c
*   Revised:        $Date: 2014-06-26 12:57:28 +0200 (to, 26 jun 2014) $
*   Revision:       $Revision: 122 $
*
*   Description:    Generic Broadcaster example code built on top of the miniBLE API.
*                   This is an example of how to use the miniBLE API to enable a
*                   non-connectable BLE broadcaster application. The application
*                   will transmit ADV_NONCONN_IND broadcast packets on all three
*                   broadcast channels (channels 37, 38 and 39).
*
*
*                   If used with the CC2543EM on the SMartRF05EB, then Button push on
*                   S1 (SmarRF05EB) is implemented to give a simple example on how to
*                   use it. The functionality must be initialized in this file while
*                   the button interrupt service routine is in the application.c file.
*
*
*   note:           Set the following "Defined Symbols" in the project options to
*                   enable/disable features in the application:
*
*                   Define "TX_INTERVAL" if you want to add a 9 ms delay in between
*                   every transmitted packet to reduce the peak current consumption
*                   drawn from the battery. If not defined then up to three
*                   broadcast packets will be transmitted back2back if all channels
*                   are enabled (channel 37, 38 and 39).
*
*                   Define "SMARTRF05EB" when using the CC2543EM + SmartRF05EB
*                   (rev. 1.8.1) plattform. In this application some debug
*                   information will then also be printed on the LCDI  LCD.
*
*                   Write "POWER_SAVING" to enable power saving. If disabled the
*                   device will still halt the CPU (power mode 0) when there is no
*                   MCU activity.
*
*
*  Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
*
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************/
/******************************************************************************
* INCLUDES
*/
#include "GenericBroadcaster_Main.h"
#include "application.h"
#include "miniBLE.h"
#include "miniBLE_defs.h"
#include "clock.h"
#include "hal_mcu.h"
#if(DTM)
#include "miniBLE_dtm.h"
#endif
#if(SMARTRF05EB)
#include "hal_board.h"
#include "hal_button.h"
#include "hal_led.h"
#include "util_lcd.h"
#include "hal_digio.h"
#endif



/*******************************************************************************
* GLOBAL VARIABLES
*/
uint8 events = 0;

/*  Arbitrary manufacture specific payload used for example application:
byte[0]:    Manufacture specific type.
byte[1]:    Texas Instruments Company Identifier Low Byte.
byte[2]:    Texas Instruments Company Identifier High Byte.
byte[3-26]: Jibberish.
*/
uint8 AdvData[27] =
{
    0xFF, 0x0D, 0x00, 0xBA, 0xD0, 0xDE, 0xED, 0x00, 0xDE, 0xAD, 0x00, 0xBE, 0xEF,
    0x00, 0xC0, 0x0F, 0xFE, 0x00, 0xF0, 0x0D, 0x00, 0xC0, 0xDC, 0xAB, 0x00, 0xBA, 0xBE
};


/*******************************************************************************
* LOCAL FUNCTIONS
*/


/*******************************************************************************
* @fn          main
*
* @brief       Main program
*
* @param       void
*
* @return      void
*/
void main(void) {
#if(SMARTRF05EB)
    /*  Initialize system clock source to 32 MHz Xtal.
    *   Initialize low speed (sleep timer) clock source to 32 kHz RCOSC.
    *   Set I/O ports and pheripherals(LCD). */
    halBoardInit();
    halLcdClear();
#if(DTM)
    halLcdWriteLine(1, "Direct Test Mode");
#else
    halLcdWriteLine(1, "2543 Broadcaster");
#endif
    halLcdWriteLine(2, " TI miniBLE ");
    halLcdWriteLine(3, "Start -> B1 (S1)");
    // Wait for user to press S1 to start BLE broadcast.
    while(halButtonPushed() != HAL_BUTTON_1);
#else
    /*  Initialize system clock source to 32 MHz Xtal.
    *   Initialize low speed (sleep timer) clock source to 32 kHz RCOSC. */
    halMcuInit();
#endif
#if(DTM)
    halSetupDTMTestPort();
    halStartDTM();
#endif

#if(SMARTRF05EB)
    // Init button S1 functionality (interrupt function given in application.c)
    initButtonS1();
#endif

    /* Initialize the miniBLE  device for broadcast operation. This will
    initialize the radio and setup the device address. */
    miniBleInit();

    // Set output power to 0 dBm.
    miniBleSetOutputPower(PLUS_0_DBM);

    // Set advertisement and data acquisition interval.
    miniBleSetTaskInterval(TASK_ADV, INTERVAL_1_SEC, SCHEDULE_LATER);
    miniBleSetTaskInterval(TASK_DAQ, INTERVAL_100_mSEC, SCHEDULE_LATER);
    miniBleSetTaskInterval(TASK_CAL, INTERVAL_500_mSEC, SCHEDULE_LATER);

    // Load test payload.
    miniBleSetAdvertisingData(AdvData, MAX_PAYLOAD_SIZE);

    // Enable all three broadcast channels.
    miniBleSetAdvertisingChannels(CHANNEL_37 | CHANNEL_38 | CHANNEL_39);

    /* Start all tasks. The task will actually begin after these commands are issued. */
    miniBleEnableTask(TASK_ADV);        // Start advertisment.
    miniBleEnableTask(TASK_DAQ);        // Start data-aquisition.
    miniBleEnableTask(TASK_CAL);        // Start calibration at given interval.


    while(1) {
        /*  miniBleWaitForNextEvent() must be called in an infinite while loop
        *   for correct operation. The return value indicates which tasks that
        *   has triggered in the sscheduler. Any processing done here
        *   (main context) can be interrupted by the schduler at any time.
        *
        *   Only very short critical sections (< 10 us or more) can be
        *   implemented, otherwise there is a risk of disrupting the
        *   advertisement interval timing. */

        // Wait for next event and enter power mode.
        events = miniBleWaitForNextEvent();

        if(events&BV(TASK_DAQ)) {
            dataAcqusition();
        }
        if(events&BV(TASK_ADV)) {
            /*  Code can be entered here if the user wants to do any processing
            *   in main context when an advertisment event occurs. Note that there
            *   might be some delay from when the event actually starts until
            *   the code execution reaches this point. */
#if(DEBUG)
            asm("NOP");
#endif
        }
        if(events&BV(TASK_CAL)) {
            // See comments inside if statement for TASK_ADV above.
#if(DEBUG)
            asm("NOP");
#endif
        }
    }
}