/*******************************************************************************
*   Filename:       broadcaster_main.c
*   Revised:        $Date: 2014-06-26 12:57:28 +0200 (to, 26 jun 2014) $
*   Revision:       $Revision: 122 $
*
*   Description:    Weather station broadcaster example code built on top of the
*                   miniBLE API. This is an example of how to use the miniBLE API
*                   to enable a non-connectable BLE broadcaster application. The
*                   application will transmit ADV_NONCONN_IND broadcast packets on
*                   all three broadcast channels (channels 37, 38 and 39).
*
*
*   note:           Set the following "Defined Symbols" in the project options to
*                   enable/disable features in the application:
*
*                   Define "TX_INTERVAL" if you want to add a 9 ms delay in between
*                   every transmitted packet to reduce the peak current consumption
*                   drawn from the battery. If not defined then up to three
*                   broadcast packets will be transmitted back2back if all channels
*                   are enabled (channel 37, 38 and 39).
*
*                   Write "POWER_SAVING" to enable power saving. If disabled the
*                   device will still halt the CPU (power mode 0) when there is no
*                   MCU activity.
*
*
*  Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
*
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************/


/******************************************************************************
* INCLUDES
*/
#include "broadcaster_main.h"
#include "weather_station.h"
#include "miniBLE.h"
#include "miniBLE_defs.h"
#include "clock.h"
#include "hal_mcu.h"
#include "hal_types.h"
#include "hal_int.h"
#include "hal_sensor.h"
#include "hal_acc.h"
#include "hal_humi.h"
#include "hal_bar.h"
#include "string.h"
#include "hal_battery.h"
#if(DTM)
#include "miniBLE_dtm.h"
#endif


/******************************************************************************
* GLOBAL VARIABLES
*/
uint8 events;


/******************************************************************************
* LOCAL FUNCTIONS
*/
/******************************************************************************
* @fn          main
*
* @brief       Main program
*
* @param       void
*
* @return      void
*/
void main(void) {

    /*  Initialize system clock source to 32 MHz Xtal.
    *   Initialize low speed (sleep timer) clock source to 32 kHz RCOSC. */
    halMcuInit();

    /* Initialize the miniBLE  device for broadcast operation. This will
    initialize the radio and setup the device address. */
    miniBleInit();

    // Wait to stabilize voltage after inserting battery ?? Needed ??
    halMcuWaitMs(100);

    // Initialize application.
    weatherStationInit();

    // Set output power to 0 dBm (reduce peak power consumption).
    miniBleSetOutputPower(PLUS_0_DBM);

    // Set advertisement and data acquisition interval.
    miniBleSetTaskInterval(TASK_ADV, INTERVAL_1_SEC, SCHEDULE_LATER);
    miniBleSetTaskInterval(TASK_DAQ, INTERVAL_1_SEC, SCHEDULE_LATER);
    miniBleSetTaskInterval(TASK_CAL, INTERVAL_500_mSEC, SCHEDULE_LATER);

    // Load empty payload for initial broadcast.
    miniBleSetAdvertisingData(AdvData, 0);

    // Enable all three broadcast channels.
    miniBleSetAdvertisingChannels(CHANNEL_37 | CHANNEL_38 | CHANNEL_39);

    // Start all tasks.
    miniBleEnableTask(TASK_ADV);        // Start advertisment.
    miniBleEnableTask(TASK_DAQ);        // Start data-aquisition.
    miniBleEnableTask(TASK_CAL);        // Start calibration at given interval.

    while(1) {
        /*  miniBleWaitForNextEvent() must be called in an infinite while loop
        *   for correct operation. The return value indicates which tasks that
        *   has triggered in the sscheduler. Any processing done here
        *   (main context) can be interrupted by the schduler at any time.
        *
        *   Only very short critical sections (< 10 us or more) can be
        *   implemented, otherwise there is a risk of disrupting the
        *   advertisement interval timing. */

        // Wait for next event and enter power mode.
        events = miniBleWaitForNextEvent();

        if(events&BV(TASK_DAQ)) {
            dataAcqusition();
        }
        if(events&BV(TASK_ADV)) {
            /*  Code can be entered here if the user wants to do any processing
            *   in main context when an advertisment event occurs. Note that there
            *   might be some delay from when the event actually starts until
            *   the code execution reaches this point. */
#if(DEBUG)
            asm("NOP");
#endif
        }
        if(events&BV(TASK_CAL)) {
            // See comments inside if statement for TASK_ADV above.
#if(DEBUG)
            asm("NOP");
#endif
        }
    }
}