/******************************************************************************
*   Filename:       hal_acc.c
*   Revised:        $Date: 2014-06-17 14:50:05 +0200 (ti, 17 jun 2014) $
*   Revision:       $Revision: 114 $
*
*   Description:    Driver for the Kionix KXTI9 Accelerometer.
*
*
*  Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
*
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************/


/******************************************************************************
* INCLUDES
*/
#include "hal_acc.h"
#include "hal_sensor.h"
#include "hal_i2c.h"
#include "hal_defs.h"


/******************************************************************************
* DEFINES
*/
// Sensor I2C address
#define HAL_KXTI9_I2C_ADDRESS          0x0F

// KXTI9 register addresses
#define ACC_REG_ADDR_XOUT_HPF_L        0x00 // R
#define ACC_REG_ADDR_XOUT_HPF_H        0x01 // R
#define ACC_REG_ADDR_YOUT_HPF_L        0x02 // R
#define ACC_REG_ADDR_YOUT_HPF_H        0x03 // R
#define ACC_REG_ADDR_ZOUT_HPF_L        0x04 // R
#define ACC_REG_ADDR_ZOUT_HPF_H        0x05 // R
#define ACC_REG_ADDR_XOUT_L            0x06 // R
#define ACC_REG_ADDR_XOUT_H            0x07 // R
#define ACC_REG_ADDR_YOUT_L            0x08 // R
#define ACC_REG_ADDR_YOUT_H            0x09 // R
#define ACC_REG_ADDR_ZOUT_L            0x0A // R
#define ACC_REG_ADDR_ZOUT_H            0x0B // R
#define ACC_REG_ADDR_DCST_RESP         0x0C // R
#define ACC_REG_ADDR_WHO_AM_I          0x0F // R
#define ACC_REG_ADDR_TILT_POS_CUR      0x10 // R
#define ACC_REG_ADDR_TILT_POS_PRE      0x11 // R

#define ACC_REG_ADDR_INT_SRC_REG1      0x15 // R
#define ACC_REG_ADDR_INT_SRC_REG2      0x16 // R
#define ACC_REG_ADDR_STATUS_REG        0x18 // R
#define ACC_REG_ADDR_INT_REL           0x1A // R

#define ACC_REG_ADDR_CTRL_REG1         0x1B // R/W
#define ACC_REG_ADDR_CTRL_REG2         0x1C // R/W
#define ACC_REG_ADDR_CTRL_REG3         0x1D // R/W

#define ACC_REG_ADDR_INT_CTRL_REG1     0x1E // R/W
#define ACC_REG_ADDR_INT_CTRL_REG2     0x1F // R/W
#define ACC_REG_ADDR_INT_CTRL_REG3     0x20 // R/W
#define ACC_REG_ADDR_DATA_CTRL_REG     0x21 // R/W

#define ACC_REG_ADDR_TILT_TIMER        0x28 // R/W
#define ACC_REG_ADDR_WUF_TIMER         0x29 // R/W
#define ACC_REG_ADDR_TDT_TIMER         0x2B // R/W
#define ACC_REG_ADDR_TDT_H_THRESH      0x2C // R/W
#define ACC_REG_ADDR_TDT_L_THRESH      0x2D // R/W
#define ACC_REG_ADDR_TDT_TAP_TIMER     0x2E // R/W
#define ACC_REG_ADDR_TDT_TOTAL_TIMER   0x2F // R/W
#define ACC_REG_ADDR_TDT_LATENCY_TIMER 0x30 // R/W
#define ACC_REG_ADDR_TDT_WINDOW_TIMER  0x31 // R/W

#define ACC_REG_ADDR_BUF_CTRL1         0x32 // R/W
#define ACC_REG_ADDR_BUF_CTRL2         0x33 // R/W
#define ACC_REG_ADDR_BUF_STATUS_REG1   0x34 // R
#define ACC_REG_ADDR_BUF_STATUS_REG2   0x35 // R/W
#define ACC_REG_ADDR_BUF_CLEAR         0x36 // W

#define ACC_REG_ADDR_SELF_TEST         0x3A // R/W

#define ACC_REG_ADDR_WUF_THRESH        0x5A // R/W
#define ACC_REG_ADDR_TILT_ANGLE        0x5C // R/W
#define ACC_REG_ADDR_HYST_SET          0x5F // R/W
#define ACC_REG_ADDR_BUF_READ          0x7F // R/W

// Select register valies
#define REG_VAL_WHO_AM_I               0x08 // (data sheet says 0x04)

// CTRL1 BIT MASKS
#define ACC_REG_CTRL_PC                0x80 // Power control  '1' On    '0' Off
#define ACC_REG_CTRL_RES               0x40 // Resolution     '1' High  '0' Low
#define ACC_REG_CTRL_DRDYE             0x20 // Data Ready     '1' On    '0' Off
#define ACC_REG_CTRL_GSEL_HI           0x10 // Range     '00' +/-2g    '01' +/-4g
#define ACC_REG_CTRL_GSEL_LO           0x08 //           '10' +/-8g    '11' N/A
#define ACC_REG_CTRL_GSEL_TDTE         0x04 // Directional Tap '1' On   '0' Off
#define ACC_REG_CTRL_GSEL_WUFE         0x02 // Wake Up         '1' On   '0' Off
#define ACC_REG_CTRL_GSEL_TPE          0x01 // Tilt Position   '1' On   '0' Off

#ifdef ACC_RANGE_8G
// Range +- 8G
#define ACC_REG_CTRL_ON                ( ACC_REG_CTRL_PC | ACC_REG_CTRL_GSEL_HI)
#define ACC_REG_CTRL_OFF               ( ACC_REG_CTRL_GSEL_HI)
#else
// Range +- 2G
#define ACC_REG_CTRL_ON                ( ACC_REG_CTRL_PC )
#define ACC_REG_CTRL_OFF               ( 0 )
#endif


/*******************************************************************************
* Local Variables
*/
static uint8 accSensorConfig;
static uint8 accSensorOff;


/*******************************************************************************
* FUNCTIONS
*/
static void HalAccSelect(void);


/******************************************************************************
* @fn          HalAccInit
*
* @brief       This function initializes the HAL Accelerometer abstraction layer.
*
* @return      None.
*/
void HalAccInit(void) {
    accSensorConfig = ACC_REG_CTRL_ON;
    accSensorOff = ACC_REG_CTRL_OFF;
}


/******************************************************************************
* @fn          HalAccRead
*
* @brief       Read data from the accelerometer - X, Y, Z - 3 bytes
*
* @return      TRUE if valid data, FALSE if not
*/
bool HalAccRead(uint8 *pBuf ) {
    uint8 x, y, z;
    bool success;

    // Select this sensor
    HalAccSelect();

    HalSensorWriteReg(ACC_REG_ADDR_CTRL_REG1, &accSensorConfig, sizeof(accSensorConfig));

    // Wait for measurement ready (appx. 1.45 ms)
    ST_HAL_DELAY(180);

    // Read the three registers
    success = HalSensorReadReg( ACC_REG_ADDR_XOUT_H, &x, sizeof(x));
    if (success) {
        success = HalSensorReadReg( ACC_REG_ADDR_YOUT_H, &y, sizeof(y));
        if (success) {
            success = HalSensorReadReg( ACC_REG_ADDR_ZOUT_H, &z, sizeof(z));
        }
    }

    if (success) {
        // Valid data
        pBuf[0] = x;
        pBuf[1] = y;
        pBuf[2] = z;
    }

    HalSensorWriteReg(ACC_REG_ADDR_CTRL_REG1, &accSensorOff, sizeof(accSensorOff));

    return success;
}


/******************************************************************************
* LOCAL FUNCTIONS
*/

/******************************************************************************
* @fn          HalAccSelect
*
* @brief       Select the accelerometer on the I2C-bus
*
* @return
*/
static void HalAccSelect(void) {
  //Set up I2C that is used to communicate with SHT21
  HalI2CInit(HAL_KXTI9_I2C_ADDRESS,i2cClock_267KHZ);
}

/*  Conversion algorithm for X, Y, Z
 *  ================================
 *
float calcAccel(int8 rawX)
{
    float v;

    //-- calculate acceleration, unit G, range -2, +2
    v = (rawX * 1.0) / (256/4);

    return v;
}
*/