/************************************************************************************
*    Filename:     hal_joystick.c
*
*    Description:  HAL joystick control implementation file
*
*  Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
*
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************/

/************************************************************************************
 * INCLUDES
 */

#include <hal_board.h>
#include <hal_joystick.h>

/***********************************************************************************
* DEFINTIONs
*/

/***********************************************************************************
* CONSTANTS
*/


/***********************************************************************************
* LOCAL VARIABLES
*/

/************************************************************************************
* @fn      halJoystickInit
*
* @brief   Initializes the joystick functionality
*
* @param  void
*
* @return void
*
************************************************************************************/
void halJoystickInit(void)
{
    // Joystick push input
    MCU_IO_INPUT(HAL_BOARD_IO_JOY_MOVE_PORT, HAL_BOARD_IO_JOY_MOVE_PIN, \
        MCU_IO_TRISTATE);
}


/************************************************************************************
* @fn  halJoystickPushed
*
* @brief
*      This function detects if the joystick is being pushed. The function
*      implements software debounce. Return true only if previuosly called
*      with joystick not pushed. Return true only once each time the joystick
*      is pressed.
*
* Parameters:
*
* @param  void
*
* @return uint8
*          1: Button is being pushed
*          0: Button is not being pushed
*
******************************************************************************/
uint8 halJoystickPushed(void)
{
  uint8 active;
  active = MCU_IO_GET(HAL_BOARD_IO_JOY_MOVE_PORT, HAL_BOARD_IO_JOY_MOVE_PIN);
  
  if(active)
  {
    // Wait until joystick push is released
    HAL_DEBOUNCE(!(MCU_IO_GET(HAL_BOARD_IO_JOY_MOVE_PORT, HAL_BOARD_IO_JOY_MOVE_PIN))); 
  }
  return active;
}

/******************************************************************************
* @fn  halJoystickGetDir
*
* @brief  This function indicates the current position of the joystick. CC2533 does
*         not have any ADC and in order to keep the PER tester menu system unchanged
*         a joystick right is simulated when the joystick is moved in any direction. 
*         Otherwise joystick center is returned.
*
* Parameters:
*
* @param  void
*
* @return uint8
*	   RIGHT:   Joystick is moved
*	   CENTER:  Joystick center
*
******************************************************************************/
uint8 halJoystickGetDir(void)
{
  // On CC2533 when HAL joystick is pushed down, simulate that joystick is pressed right
  // This is done because the menu system of the PER tester application is checking
  // whether joystick is pressed left or right. In order to use the PER test application
  // without changes joystick right is simulated when joystick is pushed. 
  if(halJoystickPushed())
  {
    return HAL_JOYSTICK_EVT_RIGHT;
  }
  else
  {
    return HAL_JOYSTICK_EVT_CENTER;
  }
}

