/******************************************************************************
*   Filename:       application.c
*   Revised:        $Date: 2014-06-26 12:57:28 +0200 (to, 26 jun 2014) $
*   Revision:       $Revision: 122 $
*
*   Description:    This is the user specific application for data accquisition
*                   and/or processing. For example, functions for sensor sampling
*                   and processing of the sampled data can be made here. An example
*                   on how to use the button S1 on SmartRF05EB is also included.
*
*
*   note            The miniBLE scheduler use the sleep timer interrupt service
*                   routine (ST ISR) to process current events/tasks and
*                   schedule next events/tasks. The RF ISR is used to
*                   handle completed transmissions and set up the next.
*                   Short critical section can be implemented.
*
*
*  Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
*
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************/


/******************************************************************************
* INCLUDES
*/
#include "application.h"
#include "miniBLE.h"
#include "miniBLE_defs.h"
#if(SMARTRF05EB)
#include "hal_board.h"
#include "hal_button.h"
#include "hal_led.h"
#include "util_lcd.h"
#include "hal_digio.h"
#endif


#define UNDEF       0
#define START_ADV   1
#define STOP_ADV    2


/*******************************************************************************
* GLOBAL VARIABLES
*/
#if(SMARTRF05EB)
uint32 counter = 0;
uint8 togglebit = 0;
uint8 buttonTrigger = 0;
const digioConfig button = {HAL_BOARD_IO_BTN_1_PORT,
                            HAL_BOARD_IO_BTN_1_PIN,
                            BV(HAL_BOARD_IO_BTN_1_PIN),
                            HAL_DIGIO_INPUT, 0};
#endif

/******************************************************************************
* LOCAL FUNCTIONS
*/

#if(SMARTRF05EB)
/******************************************************************************
* @fn       halButtonISR
*
* @brief    Interrupt service routine for button S1 on P0. Because the P0
*           interrupt routine has the same priority as the ST_ISR (where the
*           scheduler is located) the execution time must be short because it
*           cannot be preempted (interrupted) by the scheduler. Interrupts on
*           P1, P2, T1, T2, T3, T4, URX0, UTX0, ADC and ENC can have longer
*           execution time as these may be interrupted by the ST_ISR and RF_ISR.
*
*           Warning: Do NOT call miniBleWait(); here as the program will get stuck.
*
* @param    void
*
* @return   void
*/
void halButtonP0ISR(void) {
#if(DEBUG)
    asm("NOP");
#endif
    if((togglebit++)%2) {
        miniBleDisableTask(TASK_ADV);       // Stop advertisment.
        halMcuWaitMs(200);
        halLedClear(3);
    } else {
        halMcuWaitMs(200);
        miniBleEnableTask(TASK_ADV);        // Start advertisment.
        halLedSet(3);
    }
return;
}


/******************************************************************************
* @fn       initButtonS1
*
* @brief    Set up button1 on SmartRF05EB to generate an interrupt and link the
*           interrupt function halButtonP0ISR.
*
* @param    void
*
* @return   void
*/
void initButtonS1(void) {
    halDigioConfig(&button);
    halDigioIntSetEdge(&button, HAL_DIGIO_INT_RISING_EDGE);
    halDigioIntConnect(&button, &halButtonP0ISR);
    halDigioIntEnable(&button);
    return;
}
#endif


/*******************************************************************************
* @fn          dataAcqusition
*
* @brief       Function for user to implement.
*
* @param       void
*
* @return      void
*/
void dataAcqusition(void) {
#if(SMARTRF05EB)
    halLedToggle(3);            // Toggle SRF05EB LED3.
    miniBleWait(INTERVAL_50_mSEC);
    halLedToggle(3);            // Toggle SRF05EB LED3.
    miniBlePrintStatistics();   // Update LCD with statistics.
    halLcdDisplayValue(3, "DAQs: ", (uint32)++counter, " ");
#else
    asm("NOP");
#endif
    return;
}