/*******************************************************************************
*   Filename:       application.c
*   Revised:        $Date: 2014-07-09 17:39:59 +0200 (on, 09 jul 2014) $
*   Revision:       $Revision: 127 $
*
*   Description:    This is the user specific application for data accquisition
*                   and/or processing. For example, functions for sensor sampling
*                   and processing of the sampled data can be made here.
*
*
*   note            Be careful not to disable global interrupt and take great care
*                   when using interrupt service routines. The miniBLE scheduler use
*                   the sleep timer interrupt service routine (ST ISR) to process
*                   current events/tasks and schedule next events/tasks. The RF ISR is used to
*                   handle completed transmissions and set up the next.
*                   Short critical section can be implemented if used with care.
*
*
*  Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
*
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************/


/*******************************************************************************
* INCLUDES
*/
#include "ioCC2543.h"
#include "weather_station.h"
#include "miniBLE.h"
#include "miniBLE_defs.h"
#include "hal_mcu.h"
#include "hal_sensor.h"
#include "hal_acc.h"
#include "hal_humi.h"
#include "hal_bar.h"
#include "string.h"
#include "hal_battery.h"
#include "hal_defs.h"


/******************************************************************************
* DEFINES
*/
#define HIST_COUNT    12
#define AVG_COUNT      6
#define CALIBRATION_DATA_ID   0x0A;
#define MOMENTARY_DATA_ID     0xBB;
#define HISTORICAL_DATA_1_ID  0xF1  // <-1,-3> hours
#define HISTORICAL_DATA_2_ID  0xF2  // <-4,-6> hours
#define HISTORICAL_DATA_3_ID  0xF3  // <-7,-9> hours
#define HISTORICAL_DATA_4_ID  0xF4  // <-10,-12> hours
#define HISTORICAL_DATA_5_ID  0xF5  // <-10,-12> hours
#define DEVICEID 0x00; //device ID for sensor


/******************************************************************************
* TYPEDEFS
*/
typedef struct entry {
  uint16 humi;
  uint16 bTemp;
  uint16 bar;
} sensordata_t;

typedef enum {
    MOMENTARY_DATA,
    CALIBRATION_DATA,
    HISTORICAL_DATA
}wState_t;

// Global state variable.
wState_t state = MOMENTARY_DATA;


/******************************************************************************
* GLOBAL VARIABLES
*/

/*  Arbitrary manufacture specific payload used for example application:
byte[0]:    Manufacture specific type.
byte[1]:    Texas Instruments Company Identifier Low Byte.
byte[2]:    Texas Instruments Company Identifier High Byte.
byte[3]:    Device ID (TI Specific) TI Broadcaster Weather Station.
byte[3-26]: Jibberish.
*/
uint8 AdvData[27] =
{
    0xFF, 0x0D, 0x00, 0x00, 0xBA, 0xD0, 0xDE, 0xED, 0x00, 0xDE, 0xAD, 0x00, 0xBE, 0xEF,
    0x00, 0xC0, 0x0F, 0xFE, 0x00, 0xF0, 0x0D, 0x00, 0x0C, 0xAB, 0x00, 0xBA, 0xBE
};


/******************************************************************************
* LOCAL VARIABLES
*/
static sensordata_t histData[HIST_COUNT]; // Average per hour.
static uint8 aData[3];      // Data from accelerometer.
static uint8 hData[4];      // Data from hydrometer (both temp and hum.).
static uint8 bData[4];      // Data from barometer (both temp and press.).
static uint8 cData[16];     // Cal data from barometer.
static uint8 AdvDataLength;
static uint16 loopCntrHour  = 3601;  // Must be higher than 3600 to make sure a historical sample is taken upon start-up.
static uint8 specPackage    = 0;
static uint16 histDataPos   = 0;
static uint8 listFilled     = FALSE;
static uint8 battLevel      = 100;
static uint8 retValPayload = 0;


/******************************************************************************
* LOCAL FUNCTIONS
*/
/******************************************************************************
* @fn          dataAcqusition
*
* @brief       Function for user to implement.
*
* @param       void
*
* @return      void
*/
void dataAcqusition(void) {
    periodicSensorTask();
    return;
}


/******************************************************************************
* @fn       readSensors
*
* @brief    Read the sensor data and update payload buffer.
*
* @param    void
*
* @return   void
*/
#pragma inline
void periodicSensorTask(void) {
    // Read all sensors.
    readSensors();

    // Check battery level.
    battLevel = battMeasure();

    // Switch between sending immediate, calibration and historical payload.
    switch(state) {
    case MOMENTARY_DATA:
        loadImmediateData();
        if(specPackage) {
            state = HISTORICAL_DATA;
        } else {
            state = CALIBRATION_DATA;
        }
        break;
    case CALIBRATION_DATA:
        loadCalibrationData();
        state = MOMENTARY_DATA;
        specPackage = 1;
        break;
    case HISTORICAL_DATA:
        loadHistoryData();
        state = MOMENTARY_DATA;
        break;
    default:
        break;
    }


    // Update payload.
    do {
        retValPayload = miniBleSetAdvertisingData(AdvData, AdvDataLength);
    } while(retValPayload != MINIBLE_SUCCESS);


    // Save historical data every hour.
    if(loopCntrHour > 3600){  // Approximately/roughly 1 hour has elapsed.
        histData[histDataPos].humi =   BUILD_UINT16(hData[2], hData[3]);
        histData[histDataPos].bTemp =  BUILD_UINT16(bData[0], bData[1]);
        histData[histDataPos].bar =    BUILD_UINT16(bData[2], bData[3]);

        loopCntrHour = 0; // Reset second ticker
        histDataPos += 1;
        if (histDataPos > 12){
            listFilled = TRUE;
            histDataPos = 0;
        }
    }
    else {
        loopCntrHour++;
    }
}


/******************************************************************************
* @fn       loadImmediateData
*
* @brief    Load the immediate packet payload type.
*
* @param    void
*
* @return   void
*/
#pragma inline
void loadImmediateData(void) {
    AdvDataLength = 14+4;
    AdvData[4] = MOMENTARY_DATA_ID; // ID Byte.
    AdvData[5] = aData[0];          // X-axis.
    AdvData[6] = aData[1];          // Y-axis.
    AdvData[7] = aData[2];          // Z-axis.
    AdvData[8] = hData[0];          // Temperature.
    AdvData[9] = hData[1];          // Temperature.
    AdvData[10] = hData[2];         // Humidity.
    AdvData[11] = hData[3];         // Humidity.
    AdvData[12] = bData[0];         // Temperature.
    AdvData[13] = bData[1];         // Temperature.
    AdvData[14] = bData[2];         // Bar (Pressure).
    AdvData[15] = bData[3];         // Bar (Pressure).
    AdvData[16] = battLevel;        // Battery level from 0% to 100%.
    return;
}


/******************************************************************************
* @fn       loadCalibrationData
*
* @brief    Load the immediate packet payload type.
*
* @param    void
*
* @return   void
*/
#pragma inline
void loadCalibrationData(void) {
    // Send calibration data.
    AdvDataLength = 1 + 4 + sizeof(cData);
    AdvData[4] = CALIBRATION_DATA_ID;

    for(uint8 i = 0; i < sizeof(cData); i++) {
        AdvData[i+5] = cData[i];
    }
    return;
}


/******************************************************************************
* @fn       loadHistoryData
*
* @brief    Load the immediate packet payload type.
*
* @param    void
*
* @return   void
*/
#pragma inline
void loadHistoryData(void) {
    AdvData[4] = 0xF0 + specPackage; // Identifier for period.
    AdvDataLength = 1 + 4;

    // Increment start index for payload history.
    uint8 histStart = 3 * (specPackage - 1)  + 1;

    /*  Fill inn history starting with the last input (current value of
    *   histDataPos) counting down 3 times backwards in histData[] if
    *   there is any stored data. */
    for (uint8 i = histStart; i < histStart + 3; ++i ) {
        // Don't send if we don't have data.
        if (histDataPos >= i || listFilled) {
            uint8 pos = ( (histDataPos >= i) ? histDataPos : HIST_COUNT + histDataPos) - i;
            memcpy(&AdvData[AdvDataLength], &histData[pos], sizeof(sensordata_t));
            AdvDataLength += sizeof(sensordata_t);
        }
    }

    // Only advance if we have more historical data to send.
    if (specPackage < 4 && ( histDataPos >= histStart + 3 || listFilled )){
        specPackage++;
    } else {
        specPackage = 0;
    }
    return;
}


/******************************************************************************
* @fn       readSensors
*
* @brief    Function reads data from all sensors.
*
* @param    void
*
* @return   void
*/
#pragma inline
void readSensors(){
    HalAccRead(aData);                  // Read accelerometer data.
    HalHumiExecMeasurementStep(0);      // Start temp measurement.
    miniBleWait(INTERVAL_1_mSEC*15);    // Wait/Sleep for 15 ms.
    HalHumiExecMeasurementStep(1);      // Read temp data and start humi measurement.
    miniBleWait(INTERVAL_20_mSEC);      // Wait/Sleep for 20 ms.
    HalHumiExecMeasurementStep(2);      // Read humidity data.
    miniBleWait(INTERVAL_20_mSEC);      // Wait/Sleep for 20 ms.
    HalHumiReadMeasurement(hData);      // Load temp and humi data.

    /*  Read barometer data.
    *   Must be read twice due to alternate temp / bar reads. */
    HalBarStartMeasurement();
    miniBleWait(INTERVAL_5_mSEC);       // Wait/Sleep for 5 ms.
    HalBarReadMeasurement(bData);
    HalBarStartMeasurement();
    miniBleWait(INTERVAL_5_mSEC);       // Wait/Sleep for 5 ms.
    HalBarReadMeasurement(bData);
}


/******************************************************************************
* @fn       applicationInit
*
* @brief    Init the drivers for the sensors on the CC2543 Broadcaster Board.
*
* @param    void
*
* @return   void
*/
void weatherStationInit(void) {

    // Init I2C configuration.
    i2cInit();

    // Init sensor drivers.
    HalHumiInit();
    HalAccInit();
    HalBarInit();

    // Read calibration data from barometer (only has to be done once).
    HalBarReadCalibration(cData);
}


/******************************************************************************
* @fn       i2cInit
*
* @brief    Set the correct GPIO settings for the I2C module.
*
* @param    void
*
* @return   void
*/
void i2cInit(void) {
    P0SEL |= 0xC0; // Set P0_7 and P0_6 as peripheral I/O (I2C).
    P0DIR |= BV(4) | BV(3);     // P0_4 = hum-power, P0_3 = press-power.
    P1DIR |= BV(4);             // P1_4 = acc-power.
    P1INP |= BV(3);             // Tri-state acc. interrupt pin.
}