//-----------------------------------------------------------------------------
// Sensorless BLDC Motor Reference Design
//-----------------------------------------------------------------------------
// Copyright 2013 Silicon Laboratories Inc.
//
// AUTH: KAB
// DATE: 30 AUG 2006
//
// This program provides Sensorless BLDC motor control using the
// 'F310. This software is written specifically for the SBLDC
// reference design hardware. Please see the full application
// note for further information.
//
// Target: C8051F31x
//
// Tool chain: KEIL 'c' full-version required
//
// Revision 1.1 - KAB/TP (13 NOV 2013)
// - Updated Target to 'F31x
// - Added code to clear TF2H at start of ISR
// - remove static ramp loop counter and ramp variable.
//
//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f310.h>                 // SFR declarations
#include <slbdc.h>                     // macro constants
//-----------------------------------------------------------------------------
// External Public Variables
//-----------------------------------------------------------------------------
extern unsigned int Vpot;              // pot reading from T0_ISR.c
extern signed int Verror;              // error signal from T0_ISR.c
extern unsigned int NextT;             // next period to T0_ISR.c
//-----------------------------------------------------------------------------
// Public Variables
//-----------------------------------------------------------------------------
signed int Ki=0;                       // integral constant (initially zero)
signed int Kp=KPINIT;                  // proportional constant
unsigned int SpeedRPM;                 // speed in RPM for GUI
signed int Vpi;                        // output from PI controller
signed int Vout;                       // output voltage
unsigned char Speed;                   // speed for ramp controller

//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void T2_Init(void);
void T2_ISR (void);
void PI (void) ;
saturate (signed int,signed int);

//-----------------------------------------------------------------------------
// T2_Init - called from main
//-----------------------------------------------------------------------------

void T2_Init(void)
{
   CKCON   |= 0x10;                    // T2 uses SYSCLK
   TMR2RLL   = T2_RELOAD_L;            // Timer 2 Low Byte
   TMR2RLH   = T2_RELOAD_H;            // Timer 2 High Byte
}

//-----------------------------------------------------------------------------
// T2_ISR()
//-----------------------------------------------------------------------------

void T2_ISR (void) interrupt 5
{
   signed int omega;                   // used for omega calculation
   unsigned char v;                    // used for voltage calculation

   TF2H = 0;                           //

   if(Speed < (Vpot>>1))
   {
      Speed++;                      // bump up
   }
   else if (Speed > (Vpot>>1))
   {
      Speed--;                      // bump down
   }

   omega = (Speed>>1) + VMIN;          // convert 0-255 to omega

   Vout = omega;                       // copy to Vout

   PI();                               // do PI

   omega -= (Vpi>>5);                  // subtract error


   if(omega<VLIMIT)                    // fix if under limit
   {
      omega=VLIMIT;
   }
   else if(omega > (128-VLIMIT))       // fix if over limit
   {
      omega = (128-VLIMIT);
   }


   if(Vpi > 128)                       // fix if over limit
   {
      Vout = omega + VLIMIT;
   }
   else if (Vpi <-128)                 // ix if under limit
   {
      Vout = omega - VLIMIT;
   }

   if(Vout< (VMIN+VLIMIT))             // fix if under limit
   {
      Vout=(VMIN+VLIMIT);
   }
   else if(Vout > 127)                 // fix if over limit
   {
      Vout = 127;
   }

   SpeedRPM = omega * RPM_SCALE;              // calc RPM for GUI

   NextT = TRUN_TIMES_127/(unsigned char)(omega);                // calc next period


   v = Vout + 128;                     //update output voltage
   PCA0CPH0 = v;
   PCA0CPH1 = v;

}

//-----------------------------------------------------------------------------
// PI()
//-----------------------------------------------------------------------------
void PI(void)
{
   static signed int i = 0;
   signed int p,e;
   {
      e = 0;                           // clear error
      if(Kp!=0)                        // check for zero
      {
         // pre-saturate p term to 14 bits
         p = saturate (Verror, (8192/Kp));
         e += p * Kp;                  // multiply p term
      }
      if(Ki!=0)                        //check for zero
      {
         i += Verror;                  // integrate error
         i = saturate (i, (8192/Ki));  // pre-saturate i term to 14 bits
         e += i * Ki;                  // multiply i term and add (15 bits)
      }


      Vpi = e>>2;                      // shift output
   }
}

//-----------------------------------------------------------------------------
// saturate()
//-----------------------------------------------------------------------------
signed int saturate (signed int i, signed int l)
{
     if(i > l)                         // if greater than upper limit
     {
        i = +l;                        // set to upper limit
     }
     else if (i < -l)                  // if less than lower limit
     {
        i = -l;                        // set to lower limit
     }
     return i;                         // return saturated value
}