//-----------------------------------------------------------------------------
// Sensorless BLDC Motor Reference Design
//-----------------------------------------------------------------------------
// Copyright 2013 Silicon Laboratories Inc.
//
// AUTH: KAB
// DATE: 30 AUG 2006
//
// This program provides Sensorless BLDC motor control using the
// 'F310. This software is written specifically for the SBLDC
// reference design hardware. Please see the full application
// note for further information.
//
// Target: C8051F31x
//
// Tool chain: KEIL 'c' full-version required
//
// Revision 1.1 - KAB/TP (13 NOV 2013)
// - Updated Target to 'F31x
//
//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f310.h>                 // SFR declarations
#include <stdio.h>                     // printf()
#include <slbdc.h>                     // macro constants
//-----------------------------------------------------------------------------
// Function PROTOTYPES
//-----------------------------------------------------------------------------
void SYSCLK_Init (void);
void PORT_Init (void);
void PCA0_Init (void);
void ADC_Init(void);
void UART0_Init (void);
extern void StartMotor(void);          // located in T0_ISR.c
extern void T0_Init(void);             // located in T0_ISR.c
extern void T2_Init(void);             // located in T2_ISR.c
//-----------------------------------------------------------------------------
// External Public Variables
//-----------------------------------------------------------------------------
// T0_ISR Variables accessed by GUI
extern unsigned char Status;           // Motor State Variable
extern unsigned int Imotor;            // Motor current
extern unsigned int Vpot;              // Speed Pot Voltage
extern signed int Vemf;                // back EMF magnitude
extern signed int Verror;              // error voltage from midpoint
extern bit Stall;                      // Stall detection flag
extern bit OverCurrent;                // Over current flag
// T2_ISR Variables accessed by GUI
extern signed int Kp;                  // Proportional Constant
extern signed int Ki;                  // Integral Constant
extern signed int Vpi;                 // output from PI controller
extern unsigned int Vout;              // Output Voltage
extern unsigned int SpeedRPM;          // Motor speed in RPM

//-----------------------------------------------------------------------------
// Global Local Variables
//-----------------------------------------------------------------------------

sbit Start = P0^0;
sbit Stop = P0^1;

//-----------------------------------------------------------------------------
// MAIN Routine
//-----------------------------------------------------------------------------
void main (void)
{
   char theChar;

   PCA0MD &= ~0x40;                    // Disable Watchdog Timer

   SYSCLK_Init ();
   PORT_Init();                        // initialize system clock
   PCA0_Init();
   T0_Init ();
   T2_Init ();
   UART0_Init ();                      // initialize UART
   ADC_Init();
   EA = 1;                             // enable global interrupts
   printf("\r\nreset...\r\n");
   while(1)
   {
      printf("push start button...\r\n");
      while(Start==OFF);               // wait for start
      while(Start==ON);                // wait for release
      StartMotor();                    // start motor
      printf("aligning...\r\n");
      while(Status!=START);            //wait for run
      printf("starting...\r\n");
      while(Status!=RUN);              //wait for run
      printf("running...\r\n");
      while(Status==RUN)
      {
         if (RI0)                      // check for input (non-blocking)
         {
            theChar = getkey();
            while (theChar<'a' || theChar>'z')
               theChar = getkey();     // ignore non-alpha
            switch(theChar)            // parse theChar
            {
               case 'p':               // set Kp
                  printf("Kp?\r\n");
                  scanf("%d",&Kp);
                  printf("\r\nKp=%d\r\n",Kp);
                  break;
               case 'i':               // set Ki
                  printf("Ki?\r\n");
                  scanf("%d",&Ki);
                  printf("\r\nKi=%d\r\n",Ki);
                  break;
               case 's':               // display Status
                  printf("Speed=%u\r\n",SpeedRPM);
                  printf("Vout=%u\r\n",Vout);
                  printf("Verror=%d\r\n",Verror);
                  printf("Imotor=%u\r\n",Imotor);
                  printf("Vemf=%d\r\n",Vemf);
                  printf("Vpi=%d\r\n",Vpi);
                  printf("Vpot=%u\r\n",Vpot);
                  printf("Kp=%d\r\n",Kp);
                  printf("Ki=%d\r\n",Ki);
                  printf("\r\n");
                 break;
               default:
                  printf("Error\r\n");
            } //end switch
         } //end if
         if(Stall)                     // check stall flag
         {
            printf("Stall!\r\n");
         }
         if(Stop==ON)                  // check Stop button
         {
            Status = STOP;
            printf("Stopping Motor!\r\n");
            while(Stop==ON);           // wait for release
         }
      }
      if(OverCurrent)               // check over current flag
      {
         printf("OverCurrent!\r\n");
      }

   }
}

//-----------------------------------------------------------------------------
// SYSCLK_Init
//-----------------------------------------------------------------------------

void SYSCLK_Init (void)
{
   OSCICN = 0x83;                      // configure for 24.5 MHz
}
//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------

void PORT_Init (void)
{
   // P0.0 = Run,       Skipped, Open-Drain Output/Input
   // P0.1 = Reverse    Skipped, Open-Drain Output/Input
   // P0.2 = LED1       Skipped, Push-Pull Output
   // P0.3 = LED1       Skipped, Push-Pull Output
   // P0.4 = Txd        UART, Push-Pull Output
   // P0.5 = Rxd        UART, Open-Drain Output/Input
   // P0.6 = NC         Skipped, Open-Drain Output/Input
   // P0.7 = NC         Skipped, Open-Drain Output/Input
   // P1.0 = Abottom    PCA, Push-Pull Output
   // P1.1 = Atop       PCA, Push-Pull Output
   // P1.2 = Bbottom    PCA, Push-Pull Output
   // P1.3 = Btop       PCA, Push-Pull Output
   // P1.4 = Cbotton    PCA, Push-Pull Output
   // P1.5 = Ctop       PCA, Push-Pull Output
   // P1.6 = NC         Skipped, Open-Drain Output/Input
   // P1.7 = NC         Skipped, Open-Drain Output/Input
   // Port 2
   // P2.0 = VI         Skipped, Analog Input
   // P2.1 = VO         Skipped, Analog Input
   // P2.2 = VA         Skipped, Analog Input
   // P2.3 = VB         Skipped, Analog Input
   // P2.4 = VC         Skipped, Analog Input
   // P2.5 = VM         Skipped, Analog Input
   // P2.6 = Pot        Skipped, Analog Input
   // P2.7 = NC         Skipped, Analog Input


   XBR0 = 0x01;                        // Enable UART on Crossbar
   XBR1 = 0x02;                        // Enable CEX0,CEX1 on Crossbar
   P0MDOUT = 0x1C;                     // P0.2, P0.3, & P0.4 are outputs
   P1MDOUT = 077;                      // enable motor outputs (octal)
   P2MDIN = 0x00;                      // all P2 pins are Analog inputs
   P0SKIP = ~0x30;                     // Skip all, except UART
   P1SKIP = 071;                       // initial PSKIP pattern (octal)
   P2SKIP = 0x0F;                      // Skip all P2 Pins
   XBR1 |= 0x40;                       // enable crossbar
   P1    = 0xff;                       // P1 all high
}
//-----------------------------------------------------------------------------
// PCA0_Init
//-----------------------------------------------------------------------------
void PCA0_Init (void)
{

   PCA0MD = 0x02;                      // PCA uses sysclk/4, no CF int
   PCA0CPL0 = 0x00;                    // clear mode, pin high
   PCA0CPL1 = 0x00;                    // clear mode, pin high
   PCA0L = 0x00;                       // reset the timer
   PCA0H = 0x00;                       // reset the timer
   PCA0CPH0 = 0x00;                    // init to 0%
   PCA0CPH1 = 0x00;                    // init to 0%
   CR = 1;                             // START PCA0 timer
}
//-----------------------------------------------------------------------------
// ADC0_Init
//-----------------------------------------------------------------------------
void ADC_Init()
{
   AMX0P = 0x09;                       // config for motor current
   AMX0N = 0xFF;                       // single ended
   ADC0CF = 0x40;                      // SARCLK 272222,
   ADC0CN = 0x80;                      // initiate on AD0BUSY
   REF0CN    = 0x08;                   // use vdd for reference
}
//-----------------------------------------------------------------------------
// UART0_Init
//-----------------------------------------------------------------------------

void UART0_Init (void)
{
   SCON0 = 0x10;                       // enable receiver
   TMOD  &= ~0x30;                     // clear T1 mode
   TMOD |= 0x20;                       // T1 mode 2
   CKCON |= 0x08;                      // T1 uses SYSCLK
   TH1 = 0x96;                         // fixed 115200 baud
   TL1 = TH1;                          // init Timer1
   TR1 = 1;                            // START Timer1
   TI0 = 1;                            // Indicate TX0 ready
}
