//*****************************************************************************
//
//   PLC_Main.c - This file contains the Base board and daughter card
//   initialization Code for Smart IO project based on Tiva MCU
//
// 	 Copyright (c) 2012-2013 Texas Instruments Incorporated.  All rights reserved.
//   Software License Agreement
//
//   Texas Instruments (TI) is supplying this software for use solely and
//   exclusively on TI's microcontroller products. The software is owned by
//   TI and/or its suppliers, and is protected under applicable copyright
//   laws. You may not combine this software with "viral" open-source
//   software in order to form a larger program.
//
//   THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
//   NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
//   NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
//   A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
//   CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
//   DAMAGES, FOR ANY REASON WHATSOEVER.
//
/****************************HEADER*******************************************************************
*
*  $FileName: PLC_Main.c$
*  $Version : 1.0.0.0$
*  $Date    : Jan-20-2014$
*
* * Comments:
*
******************************END*********************************************************************/


/*****************************MACROS******************************************************************/

#include "PLC_main.h"


/************************* Peripheral memory map******************************************************/

#define GPIO_PORTA_BASE         0x40004000  // GPIO Port A
#define GPIO_PORTB_BASE         0x40005000  // GPIO Port B
#define GPIO_PORTC_BASE         0x40006000  // GPIO Port C
#define GPIO_PORTD_BASE         0x40007000  // GPIO Port D
#define SSI0_BASE               0x40008000  // SSI0
#define SSI1_BASE               0x40009000  // SSI1
#define SSI2_BASE               0x4000A000  // SSI2
#define SSI3_BASE               0x4000B000  // SSI3
#define I2C0_BASE               0x40020000  // I2C0
#define I2C1_BASE               0x40021000  // I2C1
#define I2C2_BASE               0x40022000  // I2C2
#define I2C3_BASE               0x40023000  // I2C3
#define GPIO_PORTE_BASE         0x40024000  // GPIO Port E
#define GPIO_PORTF_BASE         0x40025000  // GPIO Port F
#define GPIO_PORTG_BASE         0x40026000  // GPIO Port G
#define GPIO_PORTH_BASE         0x40027000  // GPIO Port H
#define I2C4_BASE               0x400C0000  // I2C4
#define GPIO_PORTJ_BASE         0x4003D000  // GPIO Port J

//*****************************************************************************
//
// The error routine that is called if the driver library encounters an error.
//
//*****************************************************************************
#ifdef DEBUG
void
__error__(char *pcFilename, unsigned int ulLine)
{
}
#endif

/*******************************************************************************************************
	Two SSI slots are provided in the base board. Base board supports 3 types of daughter cards
	ie Analog Input, Analog output and Digital Input. The daughter cards could be inserted
	into any of the two SSI slots. Depending on the inserted daughter cards, enable appropriate
	MACROS by defining them to 1. All other MACROS have to be set as 0.

	For example Analog input module is inserted in slot1 and Analog output module is inserted in slot2,
	#define AI_MODULE_SLOT1	1
	#define AI_MODULE_SLOT2	0
	#define AO_MODULE_SLOT1	0
	#define AO_MODULE_SLOT2	1
	#define DI_MODULE_SLOT_1	0
	#define DI_MODULE_SLOT_2	0
*********************************************************************************************************/

#define AI_MODULE_SLOT1	0
#define AI_MODULE_SLOT2	1
#define AO_MODULE_SLOT1	0
#define AO_MODULE_SLOT2	0
#define DI_MODULE_SLOT_1	0
#define DI_MODULE_SLOT_2	0

#define EEPROM_READ 1

#define I2C_SLAVE1_ADDR		0x20
#define I2C_SLAVE2_ADDR		0x21
#define SSI_BITRATE			(1000000)//1000kbps

/************************
 * EXTERN VARIABLES
 ************************/

/************************
 * GLOBAL VARIABLES
 ************************/

uint32_t data = 0;
unsigned int ulDataAddrVal = 0x00;
unsigned int i=0;
int dac_device;

/******************************************************************************
 * FUNCTION DEFINITION
 *****************************************************************************/

/******************************************************************************
 *    @name        	InitAnalogInputSlot1()
 *    @Precondition
 *    				Analog input module is inserted on Slot1 and AI_MODULE_SLOT1
 *    				MACRO is defined as 1.
 *    @brief		Configures and initializes Analog input module and SSI slot1
 *    @param        None
 *    @return       None
 *****************************************************************************/
void InitAnalogInputSlot1 (void)
{
	// CPLD pins should be pulled low
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);	//CPLD Pins
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOJ);	//CPLD Pins, USB Analog
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOH);	//CPLD Pins

	GPIOPinTypeGPIOOutput(GPIO_PORTC_BASE, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7);
	GPIOPinTypeGPIOOutput(GPIO_PORTJ_BASE, GPIO_PIN_2);
	GPIOPinTypeGPIOOutput(GPIO_PORTH_BASE, GPIO_PIN_5);
	GPIOPinWrite(GPIO_PORTC_BASE, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7, 0);
	GPIOPinWrite(GPIO_PORTJ_BASE, GPIO_PIN_2, 0);
	GPIOPinWrite(GPIO_PORTH_BASE, GPIO_PIN_5, 0);


	// Enable the peripherals clock for SSI - ADC and I2C- EEPROM in ADC card
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);  	//Slot1 SPI 2,3,4,5
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOG);

	//I2c Init --- I2C 3 for slot1

	SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C3);
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOG);

	GPIOPinTypeI2CSCL(GPIO_PORTG_BASE, GPIO_PIN_0);
	GPIOPinTypeI2C(GPIO_PORTG_BASE, GPIO_PIN_1);

	GPIOPinConfigure(GPIO_PG0_I2C3SCL);
	GPIOPinConfigure(GPIO_PG1_I2C3SDA);

	I2CMasterInitExpClk(I2C3_BASE, SysCtlClockGet(), false); //ROM fn initializes faster, hence no delay required

	I2CSlaveAddressSet(I2C3_BASE, I2C_SLAVE1_ADDR, false); // Write

	// SSI Init--- SSI0 for Slot 1

	SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI0);

	GPIOPinConfigure(GPIO_PA2_SSI0CLK);
	GPIOPinConfigure(GPIO_PA3_SSI0FSS);
	GPIOPinConfigure(GPIO_PA4_SSI0RX);
	GPIOPinConfigure(GPIO_PA5_SSI0TX);

	GPIOPinTypeSSI(GPIO_PORTA_BASE, GPIO_PIN_2 | GPIO_PIN_3 | GPIO_PIN_4 | GPIO_PIN_5);

	//
	// Configure and enable the SSI port for TI master mode.  Use SSI, system
	// clock supply, master mode, 1MHz SSI frequency, and 16-bit data.
	//
	SSIClockSourceSet(SSI0_BASE, SSI_CLOCK_SYSTEM);
	SSIConfigSetExpClk(SSI0_BASE, SysCtlClockGet(), SSI_FRF_MOTO_MODE_0, SSI_MODE_MASTER, SSI_BITRATE, 16);

	// Disable SPI interrupts
	SSIIntDisable(SSI0_BASE, SSI_TXFF | SSI_RXFF | SSI_RXOR | SSI_RXTO);

	// Enable SSI
	SSIEnable(SSI0_BASE);

	#if EEPROM_READ			// Read EEPROM and check whether it is ADS8638 card. If not stay in the while loop
		I2CMasterSlaveAddrSet(I2C3_BASE, 0x50, false);
		I2CMasterDataPut(I2C3_BASE, 0x00);
		I2CMasterControl(I2C3_BASE, I2C_MASTER_CMD_SINGLE_SEND);
		while(I2CMasterBusy(I2C3_BASE));

		I2CMasterSlaveAddrSet(I2C3_BASE, 0x50, true);
		I2CMasterControl(I2C3_BASE, I2C_MASTER_CMD_SINGLE_RECEIVE);
		while(I2CMasterBusy(I2C3_BASE));
		data =I2CMasterDataGet(I2C3_BASE);
		if(data != 0x01 )
			while(1);

	#endif
	while(1)
	{
		while(SSIDataGetNonBlocking(SSI0_BASE, &data))
		{
			printf("%d\n",data);
		}
		SSIDataPut(SSI0_BASE,0x0810);	//set the control reg
		SSIDataGet(SSI0_BASE,&data);
		SSIDataPut(SSI0_BASE,0x2000);	//set the config reg
		SSIDataGet(SSI0_BASE,&data);
		SSIDataPut(SSI0_BASE,0x0900);	//reading control reg
		SSIDataGet(SSI0_BASE,&data);
		SSIDataPut(SSI0_BASE, 0x0000);	//getting digitized data from ADC
		SSIDataGet(SSI0_BASE,&data);
	}
}

/******************************************************************************
 *    @name        	InitAnalogOutputSlot1()
 *    @Precondition
 *    				Analog output module is inserted on Slot1 and AO_MODULE_SLOT1 MACRO is defined as 1.
 *    @brief		Configures and initializes Analog output module and SSI slot1
 *    @param        None
 *    @return       None
 *****************************************************************************/
void InitAnalogOutputSlot1(void)
{

	// CPLD pins should be pulled low
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);	//CPLD Pins
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOJ);	//CPLD Pins, USB Analog
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOH);	//CPLD Pins

	GPIOPinTypeGPIOOutput(GPIO_PORTC_BASE, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7);
	GPIOPinTypeGPIOOutput(GPIO_PORTJ_BASE, GPIO_PIN_2);
	GPIOPinTypeGPIOOutput(GPIO_PORTH_BASE, GPIO_PIN_5);
	GPIOPinWrite(GPIO_PORTC_BASE, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7, 0);
	GPIOPinWrite(GPIO_PORTJ_BASE, GPIO_PIN_2, 0);
	GPIOPinWrite(GPIO_PORTH_BASE, GPIO_PIN_5, 0);


	// Enable the peripherals clock for SSI - ADC and I2C- EEPROM in ADC card
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);  	//Slot1 SPI 2,3,4,5
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOG);	//slot1 I2C

	//I2c Init --- I2C 3 for slot1

	SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C3);

	GPIOPinTypeI2CSCL(GPIO_PORTG_BASE, GPIO_PIN_0);
	GPIOPinTypeI2C(GPIO_PORTG_BASE, GPIO_PIN_1);

	GPIOPinConfigure(GPIO_PG0_I2C3SCL);
	GPIOPinConfigure(GPIO_PG1_I2C3SDA);

	I2CMasterInitExpClk(I2C3_BASE, SysCtlClockGet(), false); //ROM fn initializes faster, hence no delay required

	I2CSlaveAddressSet(I2C3_BASE, I2C_SLAVE1_ADDR, false); // Write

	// SSI Init--- SSI0 for Slot 1

	SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI0);

	GPIOPinConfigure(GPIO_PA2_SSI0CLK);
	GPIOPinConfigure(GPIO_PA4_SSI0RX);
	GPIOPinConfigure(GPIO_PA5_SSI0TX);

	GPIOPinTypeSSI(GPIO_PORTA_BASE, GPIO_PIN_2 | GPIO_PIN_4 | GPIO_PIN_5);

	GPIOPinTypeGPIOOutput(GPIO_PORTA_BASE, GPIO_PIN_3);				// PA3 is the chip select signal of the SSI. Make this pin as low while communicating with DAC
	GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_3, GPIO_PIN_3);

	//
	// Configure and enable the SSI port for TI master mode.  Use SSI1, system
	// clock supply, master mode, 1MHz SSI frequency, and 8-bit data.
	//
	SSIClockSourceSet(SSI0_BASE, SSI_CLOCK_SYSTEM);
	SSIConfigSetExpClk(SSI0_BASE, SysCtlClockGet(), SSI_FRF_MOTO_MODE_0, SSI_MODE_MASTER, SSI_BITRATE, 8);

	// Enable SPI interrupts
	SSIIntDisable(SSI0_BASE, SSI_TXFF | SSI_RXFF | SSI_RXOR | SSI_RXTO);

	// Enable SSI0.
	SSIEnable(SSI0_BASE);

	#if EEPROM_READ			// Read EEPROM and check whether it is DAC card. If not stay in the while loop
		I2CMasterSlaveAddrSet(I2C3_BASE, 0x50, false);
		I2CMasterDataPut(I2C3_BASE, 0x00);
		I2CMasterControl(I2C3_BASE, I2C_MASTER_CMD_SINGLE_SEND);
		while(I2CMasterBusy(I2C3_BASE));

		I2CMasterSlaveAddrSet(I2C3_BASE, 0x50, true);
		I2CMasterControl(I2C3_BASE, I2C_MASTER_CMD_SINGLE_RECEIVE);
		while(I2CMasterBusy(I2C3_BASE));
		data =I2CMasterDataGet(I2C3_BASE);
		if(data != 0x03 )
			while(1);

	#endif


	for(dac_device =3; dac_device>=0;dac_device--)
	{
		switch(dac_device)			//	dac_device stands for which dac device it is
		{
			case 0:	i=3;
					break;
			case 1:	i=2;
					break;
			case 2:	i=1;
					break;
			case 3:	i=0;
					break;
			default:i=3;
					break;
		}

		//SPI Write has 24 bits of data. Write first 8 bits(reg address) first and then the next 16 bits(reg data)
		//Write Reg addr
		ulDataAddrVal = 0x55;					//Control_Reg_Addr
		GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_3, 0);
		SSIDataPut(SSI0_BASE, ulDataAddrVal );
		//Write Reg data
		ulDataAddrVal = 0x0008;
		SSIDataPut(SSI0_BASE, 0x00 );
		SSIDataPut(SSI0_BASE, ulDataAddrVal ); // Set Daisy Bit
		//Do a no operation if the write is to DAC1 Device
		while(i--)
		{
			SSIDataPut(SSI0_BASE, 0x00 ); // Send No operation Command
			SSIDataPut(SSI0_BASE, 0x00 ); // Send 16 bit of Dummy data
			SSIDataPut(SSI0_BASE, 0x00 );
		}
		while(SSIBusy(SSI0_BASE));
		wait(10);//Delay used to maintain CS low as per observation
		GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_3, GPIO_PIN_3);

	}

	while(1)
	{

		/**********************Set Control Register******************/
		wait(10);
		GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_3, 0);
		i=4;

		while(i--)
		{
			SSIDataPut(SSI0_BASE, 0x55 );  // Send No operation Command
			SSIDataPut(SSI0_BASE, 0x10 );  // Send 16 bit of Dummy data
			SSIDataPut(SSI0_BASE, 0x00 );
		}
		wait(10);//Delay used to maintain CS low as per observation
		GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_3, GPIO_PIN_3);

		/**********************Set Configuration Register******************/
		wait(10);
		GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_3, 0);
		i=4;
		while(i--)
		{
			SSIDataPut(SSI0_BASE, 0x57 ); // Send No operation Command
			SSIDataPut(SSI0_BASE, 0x00 ); // Send 16 bit of Dummy data
			SSIDataPut(SSI0_BASE, 0x00 );
		}
		wait(10);//Delay used to maintain CS low as per observation
		GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_3, GPIO_PIN_3);

		/**********************Set DAC output******************/
		wait(10);
		GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_3, 0);
		i=4;
		while(i--)
		{
			SSIDataPut(SSI0_BASE, 0x01 ); // Send No operation Command
			SSIDataPut(SSI0_BASE, 0xFF ); // Send 16 bit of Dummy data
			SSIDataPut(SSI0_BASE, 0xFF );
		}
		wait(10);//Delay used to maintain CS low as per observation
		GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_3, GPIO_PIN_3);
	}
}

/******************************************************************************
 *    @name        	InitAnalogInputSlot2()
 *    @Precondition
 *    				Analog input module is inserted on Slot2 and AI_MODULE_SLOT2 MACRO is defined as 1.
 *    @brief		Configures and initializes Analog output module and SSI slot2
 *    @param        None
 *    @return       None
 *****************************************************************************/
void InitAnalogInputSlot2(void)
{
	// CPLD pins should be pulled low
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);	//CPLD Pins
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOJ);	//CPLD Pins, USB Analog
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOH);	//CPLD Pins

	GPIOPinTypeGPIOOutput(GPIO_PORTC_BASE, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7);
	GPIOPinTypeGPIOOutput(GPIO_PORTJ_BASE, GPIO_PIN_2);
	GPIOPinTypeGPIOOutput(GPIO_PORTH_BASE, GPIO_PIN_5);
	GPIOPinWrite(GPIO_PORTC_BASE, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7, 0);
	GPIOPinWrite(GPIO_PORTJ_BASE, GPIO_PIN_2, 0);
	GPIOPinWrite(GPIO_PORTH_BASE, GPIO_PIN_5, 0);


	// Enable the peripherals clock for SSI - ADC and I2C- EEPROM in ADC card
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);  	// Slot 2 SPI 0,1,2,3
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOG);	//Slot 2 I2C

	//I2c Init --- I2C 4 for slot2

	SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C4);

	GPIOPinTypeI2CSCL(GPIO_PORTG_BASE, GPIO_PIN_2);
	GPIOPinTypeI2C(GPIO_PORTG_BASE, GPIO_PIN_3);

	GPIOPinConfigure(GPIO_PG2_I2C4SCL);
	GPIOPinConfigure(GPIO_PG3_I2C4SDA);

	ROM_I2CMasterInitExpClk(I2C4_BASE, ROM_SysCtlClockGet(), false); //ROM fn initializes faster, hence no delay required

	I2CSlaveAddressSet(I2C4_BASE, I2C_SLAVE2_ADDR, false); // Write

	// SSI Init--- SSI1 for Slot 2

	SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI1);

	GPIOPinConfigure(GPIO_PD0_SSI1CLK);
	GPIOPinConfigure(GPIO_PD1_SSI1FSS);
	GPIOPinConfigure(GPIO_PD2_SSI1RX);
	GPIOPinConfigure(GPIO_PD3_SSI1TX);

	GPIOPinTypeSSI(GPIO_PORTD_BASE, GPIO_PIN_0 | GPIO_PIN_1 | GPIO_PIN_2 | GPIO_PIN_3);

	//
	// Configure and enable the SSI port for TI master mode.  Use SSI1, system
	// clock supply, master mode, 1MHz SSI frequency, and 8-bit data.
	//
	SSIClockSourceSet(SSI1_BASE, SSI_CLOCK_SYSTEM);
	SSIConfigSetExpClk(SSI1_BASE, SysCtlClockGet(), SSI_FRF_MOTO_MODE_0, SSI_MODE_MASTER, SSI_BITRATE, 16);

	// Disable SPI interrupts
	SSIIntDisable(SSI1_BASE, SSI_TXFF | SSI_RXFF | SSI_RXOR | SSI_RXTO);

	// Enable the SSI1 module.
	SSIEnable(SSI1_BASE);

	#if EEPROM_READ			// Read EEPROM and check whether it is ADS8638 card. If not stay in the while loop
		I2CMasterSlaveAddrSet(I2C4_BASE, 0x50, false);
		I2CMasterDataPut(I2C4_BASE, 0x00);
		I2CMasterControl(I2C4_BASE, I2C_MASTER_CMD_SINGLE_SEND);
		while(I2CMasterBusy(I2C4_BASE));

		I2CMasterSlaveAddrSet(I2C4_BASE, 0x50, true);
		I2CMasterControl(I2C4_BASE, I2C_MASTER_CMD_SINGLE_RECEIVE);
		while(I2CMasterBusy(I2C4_BASE));
		data =I2CMasterDataGet(I2C4_BASE);
		if(data != 0x01 )
			while(1);

	#endif

	while(1)
	{
		while(SSIDataGetNonBlocking(SSI1_BASE, &data))
		{
			printf("%d\n",data);
		}
		SSIDataPut(SSI1_BASE,0x0810);
		SSIDataGet(SSI1_BASE,&data);
		SSIDataPut(SSI1_BASE,0x2000);
		SSIDataGet(SSI1_BASE,&data);
		SSIDataPut(SSI1_BASE,0x0900);
		SSIDataGet(SSI1_BASE,&data);
		SSIDataPut(SSI1_BASE, 0x0000);
		SSIDataGet(SSI1_BASE,&data);


	}
}

/******************************************************************************
 *    @name        	InitAnalogOutputSlot2()
 *    @Precondition
 *    				Analog output module is inserted on Slot2 and AO_MODULE_SLOT2
 *    				MACRO is defined as 1.
 *    @brief		Configures and initializes Analog output module and SSI slot2
 *    @param        None
 *    @return       None
 *****************************************************************************/
void InitAnalogOutputSlot2(void)
{
	// CPLD pins should be pulled low
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);	//CPLD Pins
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOJ);	//CPLD Pins, USB Analog
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOH);	//CPLD Pins

	GPIOPinTypeGPIOOutput(GPIO_PORTC_BASE, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7);
	GPIOPinTypeGPIOOutput(GPIO_PORTJ_BASE, GPIO_PIN_2);
	GPIOPinTypeGPIOOutput(GPIO_PORTH_BASE, GPIO_PIN_5);
	GPIOPinWrite(GPIO_PORTC_BASE, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7, 0);
	GPIOPinWrite(GPIO_PORTJ_BASE, GPIO_PIN_2, 0);
	GPIOPinWrite(GPIO_PORTH_BASE, GPIO_PIN_5, 0);


	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);  	// Slot 2 SPI 0,1,2,3
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOG);	//slot2 I2C

	SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C4);

	GPIOPinTypeI2CSCL(GPIO_PORTG_BASE, GPIO_PIN_2);
	GPIOPinTypeI2C(GPIO_PORTG_BASE, GPIO_PIN_3);

	GPIOPinConfigure(GPIO_PG2_I2C4SCL);
	GPIOPinConfigure(GPIO_PG3_I2C4SDA);

	ROM_I2CMasterInitExpClk(I2C4_BASE, ROM_SysCtlClockGet(), false); //ROM fn initializes faster, hence no delay required

	I2CSlaveAddressSet(I2C4_BASE, I2C_SLAVE2_ADDR, false); // Write

	// SSI Init--- SSI1 for Slot 2

	SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI1);

	GPIOPinConfigure(GPIO_PD0_SSI1CLK);
	GPIOPinConfigure(GPIO_PD2_SSI1RX);
	GPIOPinConfigure(GPIO_PD3_SSI1TX);

	GPIOPinTypeSSI(GPIO_PORTD_BASE, GPIO_PIN_0 | GPIO_PIN_2 | GPIO_PIN_3);
	GPIOPinTypeGPIOOutput(GPIO_PORTD_BASE, GPIO_PIN_1);
	GPIOPinWrite(GPIO_PORTD_BASE, GPIO_PIN_1, GPIO_PIN_1); // PD1 is the chip select signal of the SSI. Make this pin as low while communicating with DAC

	//
	// Configure and enable the SSI port for TI master mode.  Use SSI0, system
	// clock supply, master mode, 1MHz SSI frequency, and 8-bit data.
	//
	SSIClockSourceSet(SSI1_BASE, SSI_CLOCK_SYSTEM);
	SSIConfigSetExpClk(SSI1_BASE, SysCtlClockGet(), SSI_FRF_MOTO_MODE_0, SSI_MODE_MASTER, SSI_BITRATE, 8);

	// Disable SPI interrupts
	SSIIntDisable(SSI1_BASE, SSI_TXFF | SSI_RXFF | SSI_RXOR | SSI_RXTO);

	// Enable the SSI1 module.
	SSIEnable(SSI1_BASE);

	#if EEPROM_READ			// Read EEPROM and check whether it is DAC card. If not stay in the while loop
		I2CMasterSlaveAddrSet(I2C4_BASE, 0x50, false);
		I2CMasterDataPut(I2C4_BASE, 0x00);
		I2CMasterControl(I2C4_BASE, I2C_MASTER_CMD_SINGLE_SEND);
		while(I2CMasterBusy(I2C4_BASE));

		I2CMasterSlaveAddrSet(I2C4_BASE, 0x50, true);
		I2CMasterControl(I2C4_BASE, I2C_MASTER_CMD_SINGLE_RECEIVE);
		while(I2CMasterBusy(I2C4_BASE));
		data =I2CMasterDataGet(I2C4_BASE);
		if(data != 0x03 )
			while(1);

	#endif

	//After Enabling the SI set the Daisy bits in all the DAC devices.

	for(dac_device =3; dac_device>=0;dac_device--)
	{
		switch(dac_device)			//	dac_device stands for which dac device it is
		{
			case 0:	i=3;
					break;
			case 1:	i=2;
					break;
			case 2:	i=1;
					break;
			case 3:	i=0;
					break;
			default:i=3;
					break;
		}


		//SPI Write has 24 bits of data. Write first 8 bits(reg address) first and then the next 16 bits(reg data)
		//Write Reg addr

		ulDataAddrVal = 0x55;
		GPIOPinWrite(GPIO_PORTD_BASE, GPIO_PIN_1, 0);
		SSIDataPut(SSI1_BASE, ulDataAddrVal );
		//Write Reg data
		ulDataAddrVal = 0x0008;
		SSIDataPut(SSI1_BASE, 0x00 );
		SSIDataPut(SSI1_BASE, ulDataAddrVal ); // Set Daisy Bit
		//Do a no operation if the write is to DAC1 Device
		while(i--)
		{
			SSIDataPut(SSI1_BASE, 0x00 ); // Send No operation Command
			SSIDataPut(SSI1_BASE, 0x00 ); // Send 16 bit of Dummy data
			SSIDataPut(SSI1_BASE, 0x00 );
		}
		while(SSIBusy(SSI1_BASE));
		wait(10);//Delay used to maintain CS low as per observation
		GPIOPinWrite(GPIO_PORTD_BASE, GPIO_PIN_1, GPIO_PIN_1);

	}
	while(1)
	{

		/**********************Set Control Register******************/
		wait(10);
		GPIOPinWrite(GPIO_PORTD_BASE, GPIO_PIN_1, 0);
		SSIDataPut(SSI1_BASE, 0x55 ); // Send No operation Command
		SSIDataPut(SSI1_BASE, 0x10 ); // Send 16 bit of Dummy data
		SSIDataPut(SSI1_BASE, 0x00 );

		i=3;

		while(i--)
		{
			SSIDataPut(SSI1_BASE, 0x00 ); // Send No operation Command
			SSIDataPut(SSI1_BASE, 0x00 ); // Send 16 bit of Dummy data
			SSIDataPut(SSI1_BASE, 0x00 );
		}
		wait(10);//Delay used to maintain CS low as per observation
		GPIOPinWrite(GPIO_PORTD_BASE, GPIO_PIN_1, GPIO_PIN_1);

		/**********************Set Configuration Register******************/
		wait(10);
		GPIOPinWrite(GPIO_PORTD_BASE, GPIO_PIN_1, 0);
		SSIDataPut(SSI1_BASE, 0x57 ); // Send No operation Command
		SSIDataPut(SSI1_BASE, 0x00 ); // Send 16 bit of Dummy data
		SSIDataPut(SSI1_BASE, 0x00 );

		i=3;
		while(i--)
		{
			SSIDataPut(SSI1_BASE, 0x00 ); // Send No operation Command
			SSIDataPut(SSI1_BASE, 0x00 ); // Send 16 bit of Dummy data
			SSIDataPut(SSI1_BASE, 0x00 );
		}
		wait(10);//Delay used to maintain CS low as per observation
		GPIOPinWrite(GPIO_PORTD_BASE, GPIO_PIN_1, GPIO_PIN_1);

		/**********************Set DAC Voltage******************/
		wait(10);
		GPIOPinWrite(GPIO_PORTD_BASE, GPIO_PIN_1, 0);
		SSIDataPut(SSI1_BASE, 0x01 );
		SSIDataPut(SSI1_BASE, 0xFF );
		SSIDataPut(SSI1_BASE, 0xFF );

		i=3;
		while(i--)
		{
			SSIDataPut(SSI1_BASE, 0x00 ); // Send No operation Command
			SSIDataPut(SSI1_BASE, 0x00 ); // Send 16 bit of Dummy data
			SSIDataPut(SSI1_BASE, 0x00 );
		}
		wait(10);//Delay used to maintain CS low as per observation
		GPIOPinWrite(GPIO_PORTD_BASE, GPIO_PIN_1, GPIO_PIN_1);
	}
}


/******************************************************************************
 *    @name        	InitDigitalInputSlot1()
 *    @Precondition
 *    				Digital input module is inserted on Slot1 and DI_MODULE_SLOT1 MACRO is defined as 1.
 *    @brief		Configures and initializes Digital Input module and SSI slot1
 *    @param        None
 *    @return       None
 *****************************************************************************/
void InitDigitalInputSlot1(void)
{
	// CPLD pins should be pulled low
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);	//CPLD Pins
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOJ);	//CPLD Pins, USB Analog
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOH);	//CPLD Pins

	GPIOPinTypeGPIOOutput(GPIO_PORTC_BASE, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7);
	GPIOPinTypeGPIOOutput(GPIO_PORTJ_BASE, GPIO_PIN_2);
	GPIOPinTypeGPIOOutput(GPIO_PORTH_BASE, GPIO_PIN_5);
	GPIOPinWrite(GPIO_PORTC_BASE, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7, 0);
	GPIOPinWrite(GPIO_PORTJ_BASE, GPIO_PIN_2, 0);
	GPIOPinWrite(GPIO_PORTH_BASE, GPIO_PIN_5, 0);


	// Enable the peripherals clock for SSI - ADC and I2C- EEPROM in ADC card
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);  	//Slot1 SPI 2,3,4,5
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOG);	//Slot 1 I2C
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);	// CS of DI card
	GPIOPinTypeGPIOOutput(GPIO_PORTE_BASE, GPIO_PIN_2);
	GPIOPinWrite(GPIO_PORTE_BASE, GPIO_PIN_2, GPIO_PIN_2); //Set LD to high by default

	//I2c Init --- I2C 3 for slot1

	SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C3);
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOG);

	GPIOPinTypeI2CSCL(GPIO_PORTG_BASE, GPIO_PIN_0);
	GPIOPinTypeI2C(GPIO_PORTG_BASE, GPIO_PIN_1);

	GPIOPinConfigure(GPIO_PG0_I2C3SCL);
	GPIOPinConfigure(GPIO_PG1_I2C3SDA);

	I2CMasterInitExpClk(I2C3_BASE, SysCtlClockGet(), false); //ROM fn initializes faster, hence no delay required
	// SysCtlDelay(I2C_INIT_DELAY);  // delay mandatory here - otherwise portion of SlaveAddrSet() lost!

	I2CSlaveAddressSet(I2C3_BASE, I2C_SLAVE1_ADDR, false); // Write
	// Enable SSI 0 to Communicate with DI card. Make CS signal as low before communicating

	SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI0);

	GPIOPinConfigure(GPIO_PA2_SSI0CLK);
	GPIOPinConfigure(GPIO_PA4_SSI0RX);
	GPIOPinConfigure(GPIO_PA5_SSI0TX);

	GPIOPinTypeSSI(GPIO_PORTA_BASE, GPIO_PIN_2 | GPIO_PIN_4 | GPIO_PIN_5);
	GPIOPinTypeGPIOOutput(GPIO_PORTA_BASE, GPIO_PIN_3);
	GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_3, GPIO_PIN_3); //Set LD to high by default

	//
	// Configure and enable the SSI port for TI master mode.  Use SSI1, system
	// clock supply, master mode, 1MHz SSI frequency, and 8-bit data.
	//
	SSIClockSourceSet(SSI0_BASE, SSI_CLOCK_SYSTEM);
	SSIConfigSetExpClk(SSI0_BASE, SysCtlClockGet(), SSI_FRF_MOTO_MODE_0, SSI_MODE_MASTER, SSI_BITRATE, 8);

	// Enable SPI interrupts
	SSIIntDisable(SSI0_BASE, SSI_TXFF | SSI_RXFF | SSI_RXOR | SSI_RXTO);

	// Enable the SSI1 module.
	SSIEnable(SSI0_BASE);

	#if EEPROM_READ			// Read EEPROM and check whether it is DI card. If not stay in the while loop
		I2CMasterSlaveAddrSet(I2C3_BASE, 0x50, false);
		I2CMasterDataPut(I2C3_BASE, 0x00);
		I2CMasterControl(I2C3_BASE, I2C_MASTER_CMD_SINGLE_SEND);
		while(I2CMasterBusy(I2C3_BASE));

		I2CMasterSlaveAddrSet(I2C3_BASE, 0x50, true);
		I2CMasterControl(I2C3_BASE, I2C_MASTER_CMD_SINGLE_RECEIVE);
		while(I2CMasterBusy(I2C3_BASE));
		data =I2CMasterDataGet(I2C3_BASE);
		if(data != 0x04 )
			while(1);
	#endif

	while(1)
	{
		GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_3, 0);
		wait(1);
		GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_3, GPIO_PIN_3);
		wait(1);

		//Set CS Low
		GPIOPinWrite(GPIO_PORTE_BASE, GPIO_PIN_2 , 0);


		SSIDataPut(SSI0_BASE,0x00);
		while(SSIBusy(SSI0_BASE));
		SSIDataGet(SSI0_BASE,&data);

		GPIOPinWrite(GPIO_PORTE_BASE, GPIO_PIN_2 , GPIO_PIN_2);
		printf("DI data - %d\n",data);

	}
}

/******************************************************************************
 *    @name        	InitDigitalInputSlot2()
 *    @Precondition
 *    				Digital input module is inserted on Slot2 and DI_MODULE_SLOT2 MACRO is defined as 1.
 *    @brief		Configures and initializes Digital Input module and SSI slot2
 *    @param        None
 *    @return       None
 *****************************************************************************/
void InitDigitalInputSlot2(void)
{
	// CPLD pins should be pulled low
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);	//CPLD Pins
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOJ);	//CPLD Pins, USB Analog
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOH);	//CPLD Pins

	GPIOPinTypeGPIOOutput(GPIO_PORTC_BASE, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7);
	GPIOPinTypeGPIOOutput(GPIO_PORTJ_BASE, GPIO_PIN_2);
	GPIOPinTypeGPIOOutput(GPIO_PORTH_BASE, GPIO_PIN_5);
	GPIOPinWrite(GPIO_PORTC_BASE, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7, 0);
	GPIOPinWrite(GPIO_PORTJ_BASE, GPIO_PIN_2, 0);
	GPIOPinWrite(GPIO_PORTH_BASE, GPIO_PIN_5, 0);


	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);  	// Slot 2 SPI 0,1,2,3
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOG);	//Slot 1 I2C 0,1. Slot 2 I2C 2,3
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);	// CS of DI card

	SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C4);

	GPIOPinTypeI2CSCL(GPIO_PORTG_BASE, GPIO_PIN_2);
	GPIOPinTypeI2C(GPIO_PORTG_BASE, GPIO_PIN_3);

	GPIOPinConfigure(GPIO_PG2_I2C4SCL);
	GPIOPinConfigure(GPIO_PG3_I2C4SDA);

	ROM_I2CMasterInitExpClk(I2C4_BASE, ROM_SysCtlClockGet(), false); //ROM fn initializes faster, hence no delay required

	I2CSlaveAddressSet(I2C4_BASE, I2C_SLAVE2_ADDR, false); // Write

	// SSI Init--- SSI1 for Slot 2

	SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI1);

	GPIOPinConfigure(GPIO_PD0_SSI1CLK);
	GPIOPinConfigure(GPIO_PD2_SSI1RX);
	GPIOPinConfigure(GPIO_PD3_SSI1TX);

	GPIOPinTypeSSI(GPIO_PORTD_BASE, GPIO_PIN_0 | GPIO_PIN_2 | GPIO_PIN_3);
	GPIOPinTypeGPIOOutput(GPIO_PORTB_BASE, GPIO_PIN_4);
	GPIOPinWrite(GPIO_PORTB_BASE, GPIO_PIN_4, GPIO_PIN_4); // PB4 is the chip select signal of the SSI. Make this pin as low while communicating with DI card

	//
	// Configure and enable the SSI port for TI master mode.  Use SSI1, system
	// clock supply, master mode, 1MHz SSI frequency, and 8-bit data.
	//
	SSIClockSourceSet(SSI1_BASE, SSI_CLOCK_SYSTEM);
	SSIConfigSetExpClk(SSI1_BASE, SysCtlClockGet(), SSI_FRF_MOTO_MODE_0, SSI_MODE_MASTER, SSI_BITRATE, 8);

	// Disable SPI interrupts
	SSIIntDisable(SSI1_BASE, SSI_TXFF | SSI_RXFF | SSI_RXOR | SSI_RXTO);

	// Enable the SSI0 module.
	SSIEnable(SSI1_BASE);

	#if EEPROM_READ			// Read EEPROM and check whether it is DI card. If not stay in the while loop
		I2CMasterSlaveAddrSet(I2C4_BASE, 0x50, false);
		I2CMasterDataPut(I2C4_BASE, 0x00);
		I2CMasterControl(I2C4_BASE, I2C_MASTER_CMD_SINGLE_SEND);
		while(I2CMasterBusy(I2C4_BASE));

		I2CMasterSlaveAddrSet(I2C4_BASE, 0x50, true);
		I2CMasterControl(I2C4_BASE, I2C_MASTER_CMD_SINGLE_RECEIVE);
		while(I2CMasterBusy(I2C4_BASE));
		data =I2CMasterDataGet(I2C4_BASE);
		if(data != 0x04 )
			while(1);
	#endif

	while(1)
	{
	//	Toggle LD pin before Reading DI card
		GPIOPinWrite(GPIO_PORTD_BASE, GPIO_PIN_1, 0);
		wait(1);
		GPIOPinWrite(GPIO_PORTD_BASE, GPIO_PIN_1, GPIO_PIN_1);
		wait(1);

		//Set CS Low
		GPIOPinWrite(GPIO_PORTB_BASE, GPIO_PIN_4 , 0);

		SSIDataPut(SSI1_BASE,0x00);
		while(SSIBusy(SSI1_BASE));
		SSIDataGet(SSI1_BASE,&data);

		//Set CS High
		GPIOPinWrite(GPIO_PORTB_BASE, GPIO_PIN_4 , GPIO_PIN_4);

		printf("DI data - %d\n",data);
	}
}

/******************************************************************************
 *    @name        main
 *    @brief       Depending on MACRO settings daughter cards are initialized.
 *
 *
 *    @param       None
 *    @return      None
 *
 *    Note: 	   This code just contains configuration/ initialization functions
 *    and not operation/ application code.
 *****************************************************************************/
void main(void)
{
	msecWait(500);  //To make sure daughter cards are powered up!
    FPULazyStackingEnable();
	IntMasterDisable(); 				//Initialization should always be done with interrupts disabled
	SysClockInit(); 					//Initialize system clock.
	SysCtlPeripheralClockGating(true);	//Controls peripheral clock gating in sleep and deep-sleep mode
	IntMasterEnable();
	#if AI_MODULE_SLOT1
		InitAnalogInputSlot1();
	#endif

	#if AO_MODULE_SLOT1
		InitAnalogOutputSlot1();
	#endif

	#if AI_MODULE_SLOT2
		InitAnalogInputSlot2();
	#endif

	#if AO_MODULE_SLOT2
		InitAnalogOutputSlot2();
	#endif

	#if DI_MODULE_SLOT_1
		InitDigitalInputSlot1();
	#endif

	#if DI_MODULE_SLOT_2
		InitDigitalInputSlot2();
	#endif
}
/******************************************************************************
 *    @name        msecWait
 *    @brief       This function holds CPU execution for period defined by argument
 *    @param       time : time in milliseconds for which CPU halts its execution or
 *                        keeps on executing nop instruction
 *    @return      None
 *****************************************************************************/
void msecWait(unsigned int time)
{
	ROM_SysCtlDelay((ROM_SysCtlClockGet()/(3*1000))*time); //ROM functions are used to get accurate results
}
/******************************************************************************
 *    @name        SysClockInit
 *    @brief       Set the system clock. It is configured to generate a clock of 50 MHz using PLL and 16MHz external OSC
 *    @param       None
 *    @return      None
 *    //TBD: Change based on requirement
 *****************************************************************************/
void SysClockInit(void)
	{
 	//
    // Set the clocking to run from the PLL at 80MHz
    //
    SysCtlClockSet(SYSCTL_SYSDIV_2_5 | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN | SYSCTL_XTAL_25MHZ);
    }

/******************************************************************************
 *    @name        wait
 *    @brief       This function holds CPU execution for period defined by argument
 *    @param       time : uncalibrated number for which delay is inserted
 *    @return      None
 *****************************************************************************/
void wait(unsigned int time)
{
	ROM_SysCtlDelay(time);
}



/* EOF */
