/******************************************************************************/
//
// Software License Agreement
//
// CODE OWNERSHIP AND DISCLAIMER OF LIABILITY
// Microchip Technology Incorporated ("Microchip") retains
// all ownership and intellectual property rights in the code
// accompanying this message and in all derivatives hereto.
// You may use this code, and any derivatives created by any
// person or entity by or on your behalf, exclusively with
// Microchips proprietary products. Your acceptance and/or
// use of this code constitutes agreement to the terms and
// conditions of this notice.
// CODE ACCOMPANYING THIS MESSAGE IS SUPPLIED BY MICROCHIP "AS IS".
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING,
// BUT NOT LIMITED TO, IMPLIED WARRANTIES OF NON-INFRINGEMENT,
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO
// THIS CODE, ITS INTERACTION WITH MICROCHIPS PRODUCTS, COMBINATION
// WITH ANY OTHER PRODUCTS, OR USE IN ANY APPLICATION.
// YOU ACKNOWLEDGE AND AGREE THAT, IN NO EVENT, SHALL MICROCHIP BE
// LIABLE, WHETHER IN CONTRACT, WARRANTY, TORT (INCLUDING NEGLIGENCE
// OR BREACH OF STATUTORY DUTY), STRICT LIABILITY, INDEMNITY, CONTRIBUTION,
// OR OTHERWISE, FOR ANY INDIRECT, SPECIAL, PUNITIVE, EXEMPLARY, INCIDENTAL
// OR CONSEQUENTIAL LOSS, DAMAGE, FOR COST OR EXPENSE OF ANY KIND WHATSOEVER
// RELATED TO THE CODE, HOWSOEVER CAUSED, EVEN IF MICROCHIP HAS BEEN ADVISED
// OF THE POSSIBILITY OR THE DAMAGES ARE FORESEEABLE. TO THE FULLEST EXTENT
// ALLOWABLE BY LAW, MICROCHIP'S TOTAL LIABILITY ON ALL CLAIMS IN ANY WAY
// RELATED TO THIS CODE, SHALL NOT EXCEED THE PRICE YOU PAID DIRECTLY TO
// MICROCHIP SPECIFICALLY TO HAVE THIS CODE DEVELOPED.
// You agree that you are solely responsible for testing the code and
// determining its suitability. Microchip has no obligation to modify,
// test, certify, or support the code.
/*****************************************************************************/
#ifndef MCP802X_H_INCLUDED
  #define MCP802X_H_INCLUDED


  /* define the values for Enabling and disabling features */
  #define DISABLED 0
  #define ENABLED  1

  /*************************** DE2_SERIAL Receive Selection *********************
   * Enable / Disable DE2 Serial Receive
   *****************************************************************************/
  #define DE2_SERIAL_RECEIVE   DISABLED


  /**************** SYSTEM DEFINES REQUIRED BEFORE LIBPIC30.H ******************/
  // Configure Oscillator for PWM and PLL operation
  // FRC:
  //   FRCNOS<2:0> = 0   internal FRC, no prescale/postscale
  //   FIN = 7.37 MHz
  // Phase Locked Loop:
  //   PLLPRE = 0, PLLPOST=0
  // N1 = PLLPRE + 2 = 2
  // N2 = 2* (PLLPOST + 1) = 2 * 1 = 2
  // M = PLLDIV + 2
  //
  // PLL Constraint:
  //  120 MHz < FVCO < 340 MHz
  // Determine range of M for PLL use:
  //  FVCO = FIN * M / N1;  M = FVCO * N1 / FIN
  //  Mmin = 120MHz * N1 / FIN = 120 MHz * 2 / 7.37 MHz = 33.56 => 33
  //  Mmax = 340MHz * N1 / FIN = 340 MHz * 2 / 7.37 MHz = 92.26 => 92
  //
  // Fosc = Fin*M/(N1*N2):  M=(N1*N2)*Fosc/Fin
  // Fcy=Fosc/2
  // Tcy = 1/Fcy
  // PWMres (bits)= log2(2 * FCY / FPWM)
  // FPWMmax = (FOSC / e^RESbits) with FVCO = 339 MHz (M=92) : FOSC = PWM * e^RESbits :
  //
  // Given RESbits and FPWM:
  //    M = (N1 * N2) * FPWM * e^RESbits / FIN
  //
  //
  /******************************* PLL Selections ***************************************/
  // Fosc= Fin*M/(N1*N2)
  // Fcy=Fosc/2 = 29.48 Mhz
  // PWMres (bits)= log2(2 * FCY / FPWM)
  // FPWM <= FOSC/(e^RESbits)
  //
  //    PWM    PLL_DIV  M   Npre  Npost  FVCO     FOSC & FPLLO     Fcy          Tcy         ResBits
  //    20KHz    10    12   0      0      44 MHz   22,110,000 Hz   11,055,000   90.457 ns   7.01
  // ** 20KHz    31    33   0      0     122 MHz   60,802,500 Hz   30,401,250   32.893 ns   8.02
  //    20KHz    86    88   0      0     324 MHz  162,140,000 Hz   81,070,000   12.335 ns   9.00
  //
  //    40KHz    22    24   0      0      88 MHz   44,220,000 Hz   22,110,000   45.228 ns   7.00
  //    40KHz    63    65   0      0     240 MHz  119,762,500 Hz   59,881,250   16.700 ns   8.00
  //
  //    50KHz    28    30   0      0     111 MHz   55,275,000 Hz   27,637,500   36.183 ns   7.00
  //    50KHz    79    81   0      0     298 MHz  149,242,500 Hz   74,621,250   13.401 ns   8.00
  //
  // max resolutions when FIN = 7.37 MHz.  FOSC can be decreased by using POST SCALING (N2)
  // FPWMmax = (FOSC / e^RESbits) with FVCO = 339 MHz (M=92)
  //     7695Hz  90    92   0      0     339 MHz  169,510,000 Hz   84,755,000   11.799 ns  10
  //    20919Hz  90    92   0      0     339 MHz  169,510,000 Hz   84,755,000   11.799 ns   9
  //    56864Hz  90    92   0      0     339 MHz  169,510,000 Hz   84,755,000   11.799 ns   8
  /**************************************************************************************/

  /********************************* PLL and PWM selection ******************************/
  /* PWM Freq */
  #define FPWM 50000

  /* define PWM resolution */
  #define PWMRESBITS 8

  /* No PLL Pre/Post dividers */
  #define N_PLLPRE (0 + 2)
  #define N_PLLPOST (0 + 2)

  #if (FPWM == 20000)
    #if (PWMRESBITS == 7)
      /* use PLL divisor for 7 bit, 20 KHz PWM */
      #define PLL_DIV 10
    #elif (PWMRESBITS == 8) /* FOSC = 60,802,500 Hz */
      /* use PLL divisor for 8 bit, 20 KHz PWM */
      #define PLL_DIV 31
    #elif (PWMRESBITS == 9)
      /* use PLL divisor for 9 bit, 20 KHz PWM */
      #define PLL_DIV 86
    #else
      Error-Invalid FPWM and PWMRESBITS combination
    #endif
  #elif (FPWM == 40000)
    #if (PWMRESBITS == 7)
      /* use PLL divisor of 62 for 7 bit, 40 KHz PWM */
      #define PLL_DIV 22
    #elif (PWMRESBITS == 8)
      /* use PLL divisor of 63 for 8 bit, 40 KHz PWM */
      #define PLL_DIV 63
    #else
      Error-Invalid FPWM and PWMRESBITS combination
    #endif
  #elif (FPWM == 50000)
    #if (PWMRESBITS == 7)
      /* use PLL divisor of 62 for 7 bit, 40 KHz PWM */
      #define PLL_DIV 22
    #elif (PWMRESBITS == 8)
      /* use PLL divisor of 59 for 8 bit, 50 KHz PWM */
      #define PLL_DIV 59
    #else
      Error-Invalid FPWM and PWMRESBITS combination
    #endif
  #else
    Error-Invalid FPWM and PWMRESBITS combination
  #endif


  #define M_PLLDIV (PLL_DIV + 2)

  /* internal oscillator frequency for FRC use */
  #define FREQ_IN 7370000UL

  /* calculate oscillator frequency */
  #define FOSC ((unsigned long) ((FREQ_IN * M_PLLDIV) / (N_PLLPRE * N_PLLPOST)))

  /* calculate Instruction Cycle Frequency and Period */
  #define FCY   ((unsigned long)(FOSC / 2UL))

  /* PWM input clock prescaler (PTCON2 bits PCLKDIV<2:0>) */
  /* 0=1:1, 1=1:2, 2=1:4, 3=1:8, 4=1:16, 5=1:32, 6=1:64, 7=Reserved */
  #define PWM_INPUT_CLOCK_PRESCALER  0

  /* PWM Phase Locked Loop Oscillator Frequency */
  #define FPLLO ((unsigned long) (FOSC >> PWM_INPUT_CLOCK_PRESCALER))

  /**************************************************************************/

  /******************************* A/D Bits *********************************/
  #define ADC_BITS 10
  #define ADC_REF_MV 3300

  /* minimum ADC clock period = TAD = 76 ns => 13,157,894.740 MHz = ADC_MAX_FREQ */
  #define ADC_MAX_FREQ 13157894UL  /* 1/TAD from PIC eletrical characteristics */
  #define ADC_ADCS (int)(( FOSC / 2 / ADC_MAX_FREQ))



  /******************************** TIMERS **********************************/
  /* Timer 1 divisor bits <5:4> in T1CON register */
  /* Timer 2 divisor bits <5:4> in T2CON register */
  #define TIMER_DIVISOR_1     1
  #define TIMER_DIVISOR_8     8
  #define TIMER_DIVISOR_64   64
  #define TIMER_DIVISOR_256  256

  #define TIMER1_DIVISOR      TIMER_DIVISOR_64
  #define TIMER2_DIVISOR      TIMER_DIVISOR_64

  #define TIMER1_TIMEBASE (FCY / TIMER1_DIVISOR)
  #define TIMER2_TIMEBASE (FCY / TIMER2_DIVISOR)

  /* determine TxCON TCKS Bits <5:4> value */
  #if (TIMER1_DIVISOR == TIMER_DIVISOR_1)
    #define TIMER1_DIVISOR_TCKPS 0
  #elif (TIMER1_DIVISOR == TIMER_DIVISOR_8)
    #define TIMER1_DIVISOR_TCKPS 1
  #elif (TIMER1_DIVISOR == TIMER_DIVISOR_64)
    #define TIMER1_DIVISOR_TCKPS 2
  #elif (TIMER1_DIVISOR == TIMER_DIVISOR_256)
    #define TIMER1_DIVISOR_TCKPS 3
  #else
    Bad Data;
  #endif

  #if (TIMER2_DIVISOR == TIMER_DIVISOR_1)
    #define TIMER2_DIVISOR_TCKPS 0
  #elif (TIMER2_DIVISOR == TIMER_DIVISOR_8)
    #define TIMER2_DIVISOR_TCKPS 1
  #elif (TIMER2_DIVISOR == TIMER_DIVISOR_64)
    #define TIMER2_DIVISOR_TCKPS 2
  #elif (TIMER2_DIVISOR == TIMER_DIVISOR_256)
    #define TIMER2_DIVISOR_TCKPS 3
  #else
    Bad Data;
  #endif


  /********************** DE2 Communications Baud Rate **********************/
  #define BAUD_RATE_UL 9600UL       /* do not change */
  /**************************************************************************/

  /**************************** Hardware Mappings **********************************/
  #define NA_RA0      PORTAbits.RA0   /* AN0 - MONITOR1 */
  #define NA_RA1      PORTAbits.RA1   /* AN1 - MONITOR2 */
  #define UNUSED_RA2  PORTAbits.RA2   /* unused port bit */
  #define UNUSED_RA3  PORTAbits.RA3   /* unused port bit */
  #define SPARE_LED   PORTAbits.RA4   /* USB LED */
  #define NA_RA5      PORTAbits.RA5   /* N/A */
  #define NA_RA6      PORTAbits.RA6   /* N/A */
  #define DRIVER_CE   PORTAbits.RA7   /* driver CE pin */
  #define SW1        !PORTAbits.RA8   /* SW1 Push Button */
  #if(DRIVER == MCP8025)
    #define nFAULT_TXE PORTAbits.RA9   /* unused port bit */
  #elif (DRIVER == MCP8026)
    #define UNUSED_RA9  PORTAbits.RA9   /* unused port bit */
  #else
    #pragma error "Driver not selected"
  #endif
  #define UNUSED_RA10 PORTAbits.RA10  /* unused port bit */


  #define MONITOR3    PORTBbits.RB0   /* AN2 - MONITOR3 */
  #define IOUTB       PORTBbits.RB1   /* AN3 - IOUTB */
  #define IOUTA       PORTBbits.RB2   /* AN4 - IOUTA */
  #define MONITOR4    PORTBbits.RB3   /* AN5 - MONITOR4 */
  #define SW2        !PORTBbits.RB4   /* SW2 Push Button */
  #define PGD         PORTBbits.RB5   /* Programming Data */
  #define PGC         PORTBbits.RB6   /* Programming Clock */
  #define DE2_RX      PORTBbits.RB7   /* RX from DE2 Communications using RP39 and UART */
  #define ILIMIT      PORTBbits.RB8   /* driver ILIMIT_OUT 1= ok, 0=Current Limit or fault, open-drain */
  #define DE2_TX      PORTBbits.RB9   /* TX to DE2 Communications using RP41 and UART  */
  #define PWM3H       PORTBbits.RB10  /* PWM3H */
  #define PWM3L       PORTBbits.RB11  /* PWM3L */
  #define PWM2H       PORTBbits.RB12  /* PWM2H */
  #define PWM2L       PORTBbits.RB13  /* PWM2L */
  #define PWM1H       PORTBbits.RB14  /* PWM1H */
  #define PWM1L       PORTBbits.RB15  /* PWM1L */

  #define IOUTC       PORTCbits.RC0   /* AN6 - IOUTC */
  #define VDD_DIV16   PORTCbits.RC1   /* AN7 - VDD_DIV16 */
  #define SPEED_POT   PORTCbits.RC2   /* AN8 - Speed Pot */
  #if (DRIVER == MCP8025)
    #define MUX1        PORTCbits.RC3   /* output: Mux select signal 1 */
    #define MUX2        PORTCbits.RC4   /* output: Mux select signal 2 */
    #define ZC          PORTCbits.RC5   /* input: Zero Crossing signal */
    #define LIN_RX      PORTCbits.RC6   /* input: LIN RX port */
    #define LIN_TX      PORTCbits.RC7   /* output: LIN TX port */
  #elif (DRIVER == MCP8026)
    #define UNUSED_RC3  PORTCbits.RC3   /* unused port bit */
    #define UNUSED_RC4  PORTCbits.RC4   /* unused port bit */
    #define UNUSED_RC5  PORTCbits.RC5   /* unused port bit */
    #define UNUSED_RC6  PORTCbits.RC6   /* unused port bit */
    #define UNUSED_RC7  PORTCbits.RC7   /* unused port bit */
  #else
    #pragma error "Driver not selected"
  #endif
  #define UNUSED_RC8  PORTCbits.RC8   /* unused port bit */
  #define UNUSED_RC9  PORTCbits.RC9   /* unused port bit */



  /****************** A/D Mappings ***********************
   * RA0 = AN0 - MONITOR1
   * RA1 = AN1 - MONITOR2
   * RB0 = AN2 - MONITOR3
   * RB1 = AN3 - IOUTB
   * RB2 = AN4 - IOUTA
   * RB3 = AN5 - MONITOR4
   * RC0 = AN6 - IOUTC
   * RC1 = AN7 - VDD_DIV16
   * RC2 = AN8 - Speed Pot
   ********************************************************/

  /****************** UART Mappings ***********************
   * RP39 = RX from DE2 Communications
   * RP41 = TX to DE2 Communications
   ********************************************************/

  #define SET_CFG0_MSG    0x81
  #define GET_CFG0_MSG    0x82
  #define SET_CFG1_MSG    0x83
  #define GET_CFG1_MSG    0x84
  #define STAT0_MSG       0x85
  #define STAT1_MSG       0x86
  #define SET_CFG2_MSG    0x87
  #define GET_CFG2_MSG    0x88

  #define SET_CFG0_MSG_RSP (SET_CFG0_MSG ^ 0b11000000)
  #define GET_CFG0_MSG_RSP (GET_CFG0_MSG ^ 0b11000000)
  #define SET_CFG1_MSG_RSP (SET_CFG1_MSG ^ 0b11000000)
  #define GET_CFG1_MSG_RSP (GET_CFG1_MSG ^ 0b11000000)
  #define STAT0_MSG_RSP    (   STAT0_MSG ^ 0b11000000)
  #define STAT1_MSG_RSP    (   STAT1_MSG ^ 0b11000000)
  #define SET_CFG2_MSG_RSP (SET_CFG2_MSG ^ 0b11000000)
  #define GET_CFG2_MSG_RSP (GET_CFG2_MSG ^ 0b11000000)

  /**************** Register Mappings **********************/
  typedef struct {
    unsigned char EXTOC    : 2; // External MOSFET overcurrent value 0=0.25V, 1=0.5V, 2=0.75V, 3=1.0V
    unsigned char EXTSC    : 1; // External MOSFET Short Circuit Detection 1=Disable, 0=Enable
    unsigned char EXTUVLO  : 1; // External MOSFET Undervoltage Lockout, 1=Disable, 0=Enable

    #if (DRIVER == MCP8025)
      unsigned char NEUSIM   : 1; // Neutral Simulator 0=Disable, 1=Enable
    #elif (DRIVER == MCP8026)
      unsigned char bit4     : 1; // unused
    #else
      #pragma error "Driver not selected"
    #endif

    unsigned char SLEEP    : 1; // Sleep Mode, 0=disable, 1=enable
    unsigned char _30KPU   : 1; // 30K LIN Bus pullup Disconnect, 0=enable disconnect when CE=0, 1=enable disconnect
    unsigned char bit7     : 1; // unused
  } CFG0struct_type;

  typedef struct {
    unsigned char DACREF   : 8; // DAC Current Reference Value, (4.503V ? 0.991V)/255 = 13.77 mV/bit, 00H =0.991V
  } CFG1struct_type;

  typedef struct {
    unsigned char DRVB     : 2; // Driver Blanking Time, 0=4us, 1=2us, 2=1us, 3=500ns
    unsigned char DRVDT    : 3; // Driver Dead Time, = (DRVDT + 1) * 250 ns
    unsigned char bit5     : 1; // unused
    unsigned char bit6     : 1; // unused
    unsigned char bit7     : 1; // unused
  } CFG2struct_type;

  typedef union {
    unsigned char all;          // entire structure byte
    struct {
      unsigned char OTPW     : 1; // Overtemperature Protection Warning, 1 = Device junction temperature is > 115C
      unsigned char OTPF     : 1; // Overtemperature Protection Fault, 1 = Device junction temperature is > 160C
      unsigned char UVLOF    : 1; // Input Undervoltage Fault, 1 = VDD Input Voltage < 5.5V
      #if (DRIVER == MCP8025)
        unsigned char DOVLOF : 1; // Driver Input Overvoltage Lockout Fault, 1 = 20V<VDDH
      #elif (DRIVER == MCP8026)
        unsigned char bit3   : 1; // unused
      #else
        #pragma error "Driver not selected"
      #endif
      unsigned char OVLOF    : 1; // Input Overvoltage Lockout Fault, 1 = VDD Input Voltage > 32V
      unsigned char BIOCPW   : 1; // Buck Input Overcurrent Protection Warning, Buck input current is above 2A peak
      unsigned char BUVLOW   : 1; // Buck Undervoltage Lockout Warning, 1=Buck output is below 90% of expected value
      unsigned char BUVLOF   : 1; // Buck Undervoltage Lockout Fault, 1=Buck output is below 80% of expected value
    };
  }STAT0struct_type;

  typedef union {
    unsigned char all;          // entire structure byte
    struct {
      unsigned char bit0     : 1; // unused
      unsigned char bit1     : 1; // unused
      unsigned char XUVLOF   : 1; // External MOSFET Gate Drive Undervoltage Fault, 1 = HSx Output voltage < 8V
      unsigned char XOCPF    : 1; // External MOSFET Overcurrent Protection Fault, 1 = External MOSFET VDS > CFG0<EXTOC> value
      unsigned char BORW     : 1; // Brown-out Reset Warning, Configuration Lost, 1 = Device internal reset has occurred since last configuration message.
      unsigned char _5VUVLOF : 1; // 5V LDO Undervoltage Lockout, 1 = 5V LDO output voltage < 4.0V
      unsigned char bit6     : 1; // unused
      unsigned char bit7     : 1; // unused
    };
  } STAT1struct_type;

  typedef union {
    unsigned char     Cmd;      // Command byte
    unsigned char     Data;     // data byte
    CFG0struct_type   Config0;  // Config 0 data
    CFG1struct_type   Config1;  // Config 1 data
    CFG2struct_type   Config2;  // Config 2 data
    STAT0struct_type  Status0;  // Status 0 data
    STAT1struct_type  Status1;  // Status 1 data
  } Msg_Type;

   /********************************************************/

#endif // MCP802X_H_INCLUDED
